package ru.yandex.solomon.experiments.uranix;

import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.net.HostAndPort;

import ru.yandex.cloud.resourcemanager.GrpcResourceManagerClient;
import ru.yandex.cloud.resourcemanager.ResourceManagerClientOptions;
import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.solomon.alert.client.AlertingClient;
import ru.yandex.solomon.alert.client.grpc.AlertingCluster;
import ru.yandex.solomon.alert.client.grpc.GrpcAlertingClient;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.fake.AnonymousAuthorizer;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolver;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolverImpl;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.ProjectPermission;
import ru.yandex.solomon.gateway.api.cloud.v1.CloudAlertController;
import ru.yandex.solomon.gateway.api.cloud.v1.CloudAuthorizer;
import ru.yandex.solomon.gateway.api.cloud.v1.CloudNotificationChannelController;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.AlertDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.AlertTypeDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.AssociatedChannelDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.ChannelConfigDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.Comparison;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.EvaluationStatusDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.ListAlert;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.ListNotification;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.NotificationChannelDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.OrderDirection;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.ThresholdAlertDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.TimeAggregation;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static org.apache.commons.lang3.builder.ToStringBuilder.reflectionToString;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class CloudApiTest {

    static class ProjectDaoStub implements ProjectsDao {

        @Override
        public CompletableFuture<Boolean> insert(Project project) {
            return CompletableFuture.failedFuture(new UnsupportedOperationException());
        }

        @Override
        public CompletableFuture<Optional<Project>> findById(String id) {
            return CompletableFuture.failedFuture(new UnsupportedOperationException());
        }

        @Override
        public CompletableFuture<List<Project>> findAllNames() {
            return CompletableFuture.failedFuture(new UnsupportedOperationException());
        }

        @Override
        public CompletableFuture<PagedResult<Project>> find(
                String text,
                String abcFilter,
                String login,
                @Nullable EnumSet<ProjectPermission> filterByPermissions,
                PageOptions pageOpts) {
            return CompletableFuture.failedFuture(new UnsupportedOperationException());
        }

        @Override
        public CompletableFuture<PagedResult<Project>> findInProjects(String text, String abcFilter, Set<String> projectIds, PageOptions pageOpts) {
            return CompletableFuture.failedFuture(new UnsupportedOperationException());
        }

        @Override
        public CompletableFuture<TokenBasePage<Project>> findV3(String text, int pageSize, String pageToken) {
            return CompletableFuture.failedFuture(new UnsupportedOperationException());
        }

        @Override
        public CompletableFuture<Optional<Project>> partialUpdate(
                Project project,
                boolean canChangeOwner,
                boolean canChangeInternalOptions,
                boolean canUpdateOldFields) {
            return CompletableFuture.failedFuture(new UnsupportedOperationException());
        }

        @Override
        public CompletableFuture<Void> upsertProjects(List<Project> projects) {
            return CompletableFuture.failedFuture(new UnsupportedOperationException());
        }

        @Override
        public CompletableFuture<Boolean> deleteOne(String id, boolean skipValidation) {
            return CompletableFuture.failedFuture(new UnsupportedOperationException());
        }

        @Override
        public CompletableFuture<Boolean> exists(String id) {
            return CompletableFuture.completedFuture(true);
        }

        @Override
        public CompletableFuture<Void> createSchemaForTests() {
            return CompletableFuture.failedFuture(new UnsupportedOperationException());
        }

        @Override
        public CompletableFuture<Void> dropSchemaForTests() {
            return CompletableFuture.failedFuture(new UnsupportedOperationException());
        }
    }

    public static void main(String[] args) {
        ProjectsDao projectsDao = new ProjectDaoStub();
        AlertingCluster cluster = new AlertingCluster(List.of(HostAndPort.fromParts("localhost", 8799)), DefaultClientOptions.empty());
        AlertingClient alertingClient = new GrpcAlertingClient(cluster);
        Executor executor = Executors.newFixedThreadPool(4);
        CloudByFolderResolver resolver = new CloudByFolderResolverImpl(new GrpcResourceManagerClient(ResourceManagerClientOptions.forPreprod().withHandlerExecutor(executor)));
        CloudAuthorizer cloudAuthorizer = new CloudAuthorizer(new AnonymousAuthorizer(), Optional.of(resolver));
        CloudAlertController cloudAlertController = new CloudAlertController(projectsDao, alertingClient, cloudAuthorizer, Optional.empty(), Optional.empty());
        CloudNotificationChannelController cloudNotificationChannelController = new CloudNotificationChannelController(projectsDao, alertingClient, cloudAuthorizer, Optional.empty());

        AuthSubject me = new AuthSubject() {
            @Override
            public String getUniqueId() {
                return "uranix";
            }

            @Override
            public AuthType getAuthType() {
                return AuthType.IAM;
            }
        };


        Test test = new Test("aoe5h5pn3otb41inm3tl", me, cloudAlertController, cloudNotificationChannelController);

        //test.testCreate();
        //test.testList();
        //test.testGet();
        //test.testChannelsList();

        test.testUpdate("f1b58e37-bed7-4afe-aa57-4032b553c8b9");

        System.exit(0);
    }

    static class Test {
        private final String folderId;
        private final CloudAlertController cloudAlertController;
        private final CloudNotificationChannelController cloudNotificationChannelController;
        private final AuthSubject me;

        Test(String folderId, AuthSubject me, CloudAlertController cloudAlertController, CloudNotificationChannelController cloudNotificationChannelController) {
            this.folderId = folderId;
            this.me = me;
            this.cloudAlertController = cloudAlertController;
            this.cloudNotificationChannelController = cloudNotificationChannelController;
        }

        void testCreate() {
            AlertDto alert = new AlertDto();

            alert.windowSecs = 300L;
            alert.name = "Created by hand";

            alert.type = new AlertTypeDto();
            alert.type.thresholdAlert = new ThresholdAlertDto();
            alert.type.thresholdAlert.selectors = "{}";
            alert.type.thresholdAlert.comparison = Comparison.GT;
            alert.type.thresholdAlert.timeAggregation = TimeAggregation.AT_LEAST_ONE;
            alert.type.thresholdAlert.transformations = "derivative(input{})";
            alert.type.thresholdAlert.alarmThreshold = 100d;
            alert.type.thresholdAlert.warningThreshold = 75d;

            alert.annotations = Map.of("foo", "{{bar}}");

            ChannelConfigDto channelConfigDto = new ChannelConfigDto();
            channelConfigDto.notifyAboutStatuses = Set.of(EvaluationStatusDto.Code.ALARM, EvaluationStatusDto.Code.WARN);
            channelConfigDto.repeatDelaySecs = 0L;

            alert.channels = List.of(AssociatedChannelDto.of("nonexistent", channelConfigDto));

            AlertDto alert2 = cloudAlertController.createAlert(me, folderId, alert).join();

            System.out.println(alert);
        }

        void testUpdate(String alertId) {
            AlertDto alert = cloudAlertController.getAlert(me, folderId, alertId).join();
            System.out.println(reflectionToString(alert.channelProps.get(0)));

            alert.type.thresholdAlert.warningThreshold -= 1.0;

            AlertDto updated = cloudAlertController.updateAlert(me, folderId, alertId, alert).join();

            System.out.println(reflectionToString(updated.channelProps.get(0)));
        }

        void testList() {
            ListAlert alerts = cloudAlertController.listAlerts(me, folderId,
                "",
                null,
                null,
                null,
                null,
                OrderDirection.ASC,
                OrderDirection.ASC,
                OrderDirection.ASC,
                30, ""
            ).join();

            for (var item : alerts.items) {
                System.out.println(reflectionToString(item));
            }
        }

        void testChannelsList() {
            ListNotification notifications = cloudNotificationChannelController.listNotifications(me, folderId,
                "",
                null,
                null,
                null,
                30, ""
            ).join();

            for (var item : notifications.items) {
                System.out.println(reflectionToString(item));
            }
        }

        void testGet() {
            NotificationChannelDto notificationChannel = cloudNotificationChannelController.getNotification(me, folderId, "").join();
        }
    }
}
