package ru.yandex.solomon.experiments.uranix;

import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.mutable.MutableInt;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.discovery.cluster.ClusterMapperImpl;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.config.SolomonConfigs;
import ru.yandex.solomon.config.protobuf.frontend.TGatewayConfig;
import ru.yandex.solomon.config.thread.StubThreadPoolProvider;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagHolderStub;
import ru.yandex.solomon.gateway.api.v2.dto.data.DataResultDto;
import ru.yandex.solomon.gateway.data.DataClient;
import ru.yandex.solomon.gateway.data.DataClientMetrics;
import ru.yandex.solomon.gateway.data.DataRequest;
import ru.yandex.solomon.gateway.data.DataResponse;
import ru.yandex.solomon.gateway.data.DownsamplingOptions;
import ru.yandex.solomon.gateway.data.DownsamplingType;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.MetricsClientFactory;
import ru.yandex.solomon.util.time.Interval;

import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.solomon.experiments.uranix.pmconf.PoorManConfDb.makeConfigForItype;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class ExpressionGateway {
    private final MetricsClient metricsClient;
    private final DataClient dataClient;
    private final MetricRegistry registry;

    public static void main(String[] args) {
        //String gatewayConfig = "/home/uranix/arcadia/solomon/configs/production/gateway.conf";
        String gatewayConfig = "/home/uranix/arcadia/solomon/configs/dev/gateway.local-dp.conf";
        //String gatewayConfig = "/home/uranix/arcadia/solomon/configs/prestable/gateway.conf";

        (new ExpressionGateway(gatewayConfig)).run();
    }

    private static final String ITYPE = "yasmagent";
    /*
                    service=yasm, cluster="host_0",
                hosts="sas2-3272.search.yandex.net",
                ctype=prod,
                geo=sas,
                signal="portoinst-cpu_usage_cores_hgram"
     */
    private static final String request = """
            {
                project="yasm_yasmagent", cluster='group_*', service='yasm',
                signal="portoinst-cpu_usage_cores_hgram"
            }
            """;

    private ExpressionGateway(String gatewayConfig) {
        TGatewayConfig config = SolomonConfigs.parseConfig(gatewayConfig, TGatewayConfig.getDefaultInstance());

        ThreadPoolProvider threadPool = new StubThreadPoolProvider();

        ClusterMapper clusterMapper = new ClusterMapperImpl(config.getClustersConfigList(), DiscoveryService.async(), threadPool.getExecutorService("", ""), threadPool.getSchedulerExecutorService());
        registry = new MetricRegistry();

        SolomonConfHolder confHolder = new SolomonConfHolder();
        SolomonRawConf raw = makeConfigForItype("yasm_pull_sts1_" + ITYPE, 10, 10);

        confHolder.onConfigurationLoad(SolomonConfWithContext.create(raw));
        ClientOptionsFactory clientOptionsFactory = new ClientOptionsFactory(Optional.empty(), Optional.empty(), threadPool);

        var metricsClientFactory = new MetricsClientFactory(threadPool, registry, clusterMapper, clientOptionsFactory);
        var flagsHolder = new FeatureFlagHolderStub();
        flagsHolder.setFlag("yasm_" + ITYPE, FeatureFlag.METADATA_FROM_DATAPROXY, true);
        flagsHolder.setFlag("yasm_" + ITYPE, FeatureFlag.SERIES_FROM_DATAPROXY, true);
        metricsClient = metricsClientFactory.create(
                "expression-gateway",
                config.getMetabaseClientConfig(),
                config.getStockpileClientConfig(),
                config.getDataProxyClientConfig(),
                flagsHolder);
        dataClient = new DataClient(new DataClientMetrics(), metricsClient, clusterMapper, registry, confHolder, flagsHolder);
    }

    private void run() {
        Map<String, MutableInt> outcome = new HashMap<>();

        boolean boot = false;
        if (boot) {
            System.out.println("Waiting for the clients to boot");
            while (true) {
                var mbAvail = metricsClient.getMetabaseAvailability().getAvailability();
                var spAvail = metricsClient.getStockpileAvailability().getAvailability();
                System.out.println("MB avail: " + mbAvail + ", SP avail: " + spAvail);

                if (mbAvail < 1 || spAvail < 1) {
                    try {
                        Thread.sleep(2_000);
                    } catch (InterruptedException ignore) {
                    }
                } else {
                    break;
                }
            }
        }

        for (int tries = 0; tries < 3; tries++) {
            try {
                String program = request;
                var request = DataRequest.newBuilder()
                        .setProjectId("yasm_" + ITYPE)
                        .setProgram(program)
                        .setInterval(Interval.before(Instant.now(), Duration.ofHours(1)))
                        .setProducer(RequestProducer.STAFF)
                        .setDeadline(Instant.now().plusSeconds(90))
                        //.setOldMode(true)
                        .setDownsampling(DownsamplingOptions.newBuilder()
                                .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                                .setGridMillis(15_000)
                                .setDownsamplingFill(OperationDownsampling.FillOption.NONE)
                                .build())
                        //.setUseNewFormat(false)
                        //.setSummaryOnly(true)
                        .build();
                DataResponse response = join(dataClient.readData(request));
                var respDto = DataResultDto.fromModel(response);
                System.out.println(response.getOldModeResult());
                outcome.computeIfAbsent(response.toString(), ignore -> new MutableInt(0)).increment();
            } catch (Throwable e) {
                e = CompletableFutures.unwrapCompletionException(e);
                System.err.println(e);
                outcome.computeIfAbsent(e.getClass().getSimpleName(), ignore -> new MutableInt(0)).increment();
            }
        }
        for (var entry : outcome.entrySet()) {
            System.out.println(entry.getValue() + " : " + entry.getKey());
        }
        System.exit(0);
    }
}
