package ru.yandex.solomon.experiments.uranix;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.ParametersAreNonnullByDefault;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.discovery.cluster.ClusterMapperImpl;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.SolomonConfigs;
import ru.yandex.solomon.config.protobuf.frontend.TGatewayConfig;
import ru.yandex.solomon.config.thread.StubThreadPoolProvider;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.coremon.meta.db.ydb.YdbMetricsDaoFactory;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.main.logger.LoggerConfigurationUtils;
import ru.yandex.solomon.secrets.SecretProviders;
import ru.yandex.solomon.util.async.InFlightLimiter;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.ydb.YdbClients;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.ReadMetricsMetaRequest;
import ru.yandex.stockpile.api.ReadMetricsMetaResponse;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientFactory;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class LastTsExtractor {
    private static final ObjectMapper mapper = new ObjectMapper();

    public static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withNumRetries(5)
            .withDelay(TimeUnit.SECONDS.toMillis(30))
            .withMaxDelay(TimeUnit.MINUTES.toMillis(5));


    private final StockpileClient stockpile;
    private final YdbMetricsDaoFactory daoFactory;

    public static class Args {
        @Parameter(names = { "-i", "--num-id" }, description = "Shard numId", required = true)
        Integer numId;

        @Parameter(names = { "--config" }, description = "Path to gateway config", required = true)
        String config;

        @Parameter(names = { "--output" }, description = "Output file with dump", required = true)
        String output;

        @Parameter(names = {"-h", "--help"}, description = "Show this help", help = true)
        boolean help;
    }

    public static void main(String[] args) {
//        args = new String[] {
//                "-i", "797803362",
//                "--config", "/home/uranix/arcadia/solomon/configs/dev/last_ts_extractor.conf",
//                "--output", "/tmp/yasm_yasmsrv_group_2.dump"
//        };

        Args params = new Args();

        JCommander parser = new JCommander(params);
        parser.parse(args);

        if (args.length == 0 || params.help) {
            parser.usage();
            return;
        }

        try {
            var extractor = new LastTsExtractor(params.config);
            extractor.run(params.output, params.numId);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            System.exit(0);
        }
    }

    private LastTsExtractor(String gatewayConfig) {
        LoggerConfigurationUtils.disableLogger();
        TGatewayConfig config = SolomonConfigs.parseConfig(gatewayConfig, TGatewayConfig.getDefaultInstance());

        ThreadPoolProvider threadPool = new StubThreadPoolProvider();
        ClusterMapper clusterMapper = new ClusterMapperImpl(config.getClustersConfigList(), DiscoveryService.async(), threadPool.getExecutorService("", ""), threadPool.getSchedulerExecutorService());
        MetricRegistry registry = new MetricRegistry();

        var clientOptionsFactory = new ClientOptionsFactory(Optional.empty(), Optional.empty(), threadPool);
        var stockpileClientFactory = new StockpileClientFactory(threadPool, clusterMapper, registry, clientOptionsFactory);

        String cluster = config.getClustersConfig(0).getClusterId();
        System.out.println("Using cluster: " + cluster);

        stockpile = stockpileClientFactory.createClients("last-ts-extractor", config.getStockpileClientConfig()).get(cluster);

        var dbConfig = config.getCrossDcKikimrClientConfig();
        var ydbClients = new YdbClients("",
                dbConfig,
                threadPool,
                MetricRegistry.root(),
                Optional.empty(),
                SecretProviders.empty());
        var path = dbConfig.getSchemaRoot() + "/V1";
        daoFactory = YdbMetricsDaoFactory.forReadWrite(ydbClients.getTableClient(), path, registry);
    }

    private void sleep(long millis) {
        try {
            Thread.sleep(millis, 0);
        } catch (InterruptedException ignored) {
        }
    }

    private record LabelWithStockpileId(Labels labels, long stockpileLocalId) {}

    private static class Progress {
        private final long total;
        private final long startNanos;
        private long offset;

        public Progress(long total) {
            this.total = total;
            this.offset = 0;
            this.startNanos = System.nanoTime();
        }

        void add(long value) {
            this.offset += value;
        }

        public String toString() {
            long elapsedNanos = System.nanoTime() - startNanos;
            long remainingNanos = elapsedNanos / offset * (total - offset);
            double percent = 100d * offset / total;
            return String.format("%5.1f %% ETA %.2fm, %d / %d", percent,
                    TimeUnit.NANOSECONDS.toSeconds(remainingNanos) / 60d, offset, total);
        }
    }

    private void run(String outputFile, int numId) throws IOException {
        var dao = daoFactory.create(numId, new InterningLabelAllocator());

        final long startNanos = System.nanoTime();
        System.out.println("\nMETABASE\n");

        long totalCount = RetryCompletableFuture.runWithRetries(dao::getMetricCount, RETRY_CONFIG).join();

        AtomicLong loaded = new AtomicLong(0);
        Map<Integer, List<LabelWithStockpileId>> localIdByShardId = new ConcurrentHashMap<>();
        dao.findMetrics(metrics -> {
            for (int i = 0; i < metrics.size(); i++) {
                var metric = metrics.get(i);
                localIdByShardId.computeIfAbsent(metric.getShardId(), ignore -> new ArrayList<>())
                        .add(new LabelWithStockpileId(metric.getLabels(), metric.getLocalId()));
            }
            loaded.addAndGet(metrics.size());
            long offset = loaded.get();
            double elapsedSecs = 1e-9 * (System.nanoTime() - startNanos);
            System.out.printf("Resolving at %8.2f metric/sec -> %5.1f %% ETA %.2fm\n",
                    offset / elapsedSecs, 100d * offset / totalCount,
                    elapsedSecs / offset * (totalCount - offset) / 60d);
        }, OptionalLong.of(totalCount)).join();

        System.out.println("\nSTOCKPILE\n");
        int limitInStockpileBatch = 10_000;

        Progress progress = new Progress(totalCount);
        InFlightLimiter limiter = new InFlightLimiter(50);

        try (PrintStream ps = new PrintStream(new FileOutputStream(outputFile))) {
            for (var entry : localIdByShardId.entrySet()) {
                int shardId = entry.getKey();
                List<LabelWithStockpileId> metrics = entry.getValue();
                resolveLastTs(limiter, progress, ps, shardId, metrics, limitInStockpileBatch);
            }

            while (true) {
                int wait = limiter.getWaitingCount();
                if (wait == 0) {
                    break;
                }
                sleep(500);
            }
        }
    }

    private void resolveLastTs(InFlightLimiter limiter, Progress progress, PrintStream ps, int shardId, List<LabelWithStockpileId> metrics, int limitInBatch) throws IOException {
        int offset = 0;

        Semaphore semaphore = new Semaphore(1);

        while (offset < metrics.size()) {
            var request = ReadMetricsMetaRequest.newBuilder()
                .setShardId(shardId);

            int high = Math.min(metrics.size(), offset + limitInBatch);
            var idToLabels = new Long2ObjectOpenHashMap<Labels>();
            for (; offset < high; offset++) {
                var metric = metrics.get(offset);
                request.addLocalIds(metric.stockpileLocalId);
                idToLabels.put(metric.stockpileLocalId, metric.labels);
            }
            limiter.run(() -> readMetricsMeta(request.build())
                    .thenAccept(resp -> {
                        try {
                            semaphore.acquire();
                        } catch (InterruptedException e) {
                            throw new RuntimeException(e);
                        }
                        progress.add(resp.getMetaCount());
                        System.out.println("Reading stockpile " + progress);
                        resp.getMetaList().forEach(meta -> {
                            var row = mapper.createObjectNode();
                            row.put("__last_timestamp_ms", meta.getLastTsMillis());
                            idToLabels.get(meta.getLocalId()).forEach(label -> row.put(label.getKey(), label.getValue()));
                            try {
                                ps.println(mapper.writeValueAsString(row));
                            } catch (JsonProcessingException e) {
                                e.printStackTrace();
                            }
                        });
                        semaphore.release();
                    }));

        }
    }

    private CompletableFuture<ReadMetricsMetaResponse> readMetricsMeta(ReadMetricsMetaRequest request) {
        return RetryCompletableFuture.runWithRetries(() -> stockpile.readMetricsMeta(request)
                .thenApply(resp -> {
                    if (resp.getStatus()  != EStockpileStatusCode.OK) {
                        throw new RuntimeException(resp.getStatus().toString());
                    }
                    return resp;
                }), RETRY_CONFIG);
    }
}
