package ru.yandex.solomon.experiments.uranix;

import java.io.FileInputStream;
import java.io.IOException;
import java.time.Duration;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.ambry.dto.ListResponse;
import ru.yandex.ambry.dto.YasmAlertDto;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.protobuf.TAlertOrBuilder;
import ru.yandex.solomon.expression.analytics.Program;
import ru.yandex.solomon.expression.compile.DeprOpts;
import ru.yandex.solomon.expression.exceptions.SelException;
import ru.yandex.solomon.util.concurrent.ForkJoinPools;
import ru.yandex.solomon.yasm.alert.converter.YasmAlertConverter;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class OfflineYasmAlertConverter {
    public static void main(String[] args) {

    }

    public static void main2(String[] args) throws IOException {
        ObjectMapper mapper = new ObjectMapper();
        FileInputStream inputStream = new FileInputStream("/home/uranix/yasm_alerts/whole_list_pretty.json");
        var result = mapper.readValue(inputStream, ListResponse.class);
        List<YasmAlertDto> alerts = result.response.result;
        AtomicLong ok = new AtomicLong();
        AtomicLong error = new AtomicLong();
        AtomicLong total = new AtomicLong();
        ConcurrentHashMap<String, ConcurrentLinkedQueue<String>> errorsCount = new ConcurrentHashMap<>();
        ExecutorService pool = ForkJoinPools.newPool(2, "converter");
        long started = System.nanoTime();
        YasmAlertConverter converter = new YasmAlertConverter("yasm");
        alerts.stream()
                .map(alert -> CompletableFuture.supplyAsync(() -> {
                    //System.out.println(alert.name + ": " + alert.signal);
                    try {
                        TAlertOrBuilder alertProto = converter.convertAlert(alert);
                        String program = alertProto.getExpression().getProgram();
                        System.out.println(program);
                        Program.fromSource(program).withDeprOpts(DeprOpts.ALERTING).compile();
                        ok.incrementAndGet();
                    } catch (Throwable e) {
                        String msg = e.getClass().getSimpleName() + ": " + e.getMessage();
                        if (e instanceof SelException) {
                            System.err.println(msg);
                        }
                        error.incrementAndGet();
                        errorsCount.computeIfAbsent(msg, ignore -> new ConcurrentLinkedQueue<>()).add(alert.name);
                    }
                    long processed = total.incrementAndGet();
                    if (processed % 1000 == 0) {
                        System.out.println(String.format("%.2f%% done", 100d * processed / alerts.size()));
                    }
                    return null;
                }, pool))
                .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOfVoid))
                .join();
        long finished = System.nanoTime();
        System.out.println("Elapsed: " + Duration.ofNanos(finished - started).toMillis() + " ms");
        System.out.println("OK: " + ok.get() + ", ERROR: " + error.get());
        var entries = errorsCount.entrySet().stream()
                .sorted(Comparator.comparing(e -> -(e.getValue().size())))
                .collect(Collectors.toList());

        for (var entry : entries) {
            System.out.println(entry.getKey() + ": " + entry.getValue().size());
            for (var name : entry.getValue()) {
                System.out.println("> " + name);
            }
        }

        System.err.flush();
    }
}
