package ru.yandex.solomon.experiments.uranix;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.expression.SelLexer;
import ru.yandex.solomon.expression.analytics.Program;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Ivan Tsybulin
 *
 * Orig SelLexer
 *
 * Benchmark         Mode  Cnt  Score   Error  Units
 * LexAlerts.lexAll  avgt   10  6,153 ± 1,610   s/op
 *
 * New SelLexer
 *
 * Benchmark         Mode  Cnt  Score   Error  Units
 * LexAlerts.lexAll  avgt   10  1,571 ± 0,021   s/op
 *
 *
 * Benchmark                      Mode  Cnt   Score   Error  Units
 * ParsingBenchmark.compileBench  avgt    3  23,834 ± 4,817   s/op
 * ParsingBenchmark.lexBench      avgt    3   5,519 ± 0,529   s/op
 *
 * Benchmark                      Mode  Cnt  Score   Error  Units
 * ParsingBenchmark.compileBench  avgt    3  6,845 ± 0,941   s/op
 * ParsingBenchmark.lexBench      avgt    3  1,649 ± 0,372   s/op
 *
 */
@ParametersAreNonnullByDefault
@Fork(value = 1)
@Warmup(iterations = 2)
@Measurement(iterations = 3)
@BenchmarkMode(Mode.AverageTime)
public class ParsingBenchmark {
    private static List<String> sources = getSources();

    private static class Code {
        public String program;
    }

    private static List<SelLexer.Token> lex(List<String> sources) {
        List<SelLexer.Token> tokens = new ArrayList<>();
        for (String source : sources) {
            SelLexer lexer = new SelLexer(source);
            tokens.addAll(lexer.nextTokens());
        }
        return tokens;
    }

    private static List<Object> compile(List<String> sources) {
        List<Object> programs = new ArrayList<>(sources.size());
        for (String source : sources) {
            try {
                Instant now = Instant.parse("2019-11-01T00:00:00Z");
                Instant prev = now.minus(Duration.ofHours(1));
                programs.add(Program.fromSource(source).compile().prepare(new Interval(prev, now)).format());
            } catch (Exception e) {
                System.err.println("Bad program:\n" + source);
                e.printStackTrace();
                programs.add(e.getMessage());
            }
        }
        return programs;
    }

    @Benchmark
    public List<SelLexer.Token> lexBench() {
        return lex(sources);
    }

    @Benchmark
    public List<Object> compileBench() {
        return compile(sources);
    }

    public static void main2(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(ParsingBenchmark.class.getName())
            .detectJvmArgs()
            .jvmArgs("-Xmx1g", "-Xms1g")
            .build();

        new Runner(opt).run();
    }

    public static void main(String[] args) {
        System.out.println("done");
    }

    private static List<String> getSources() {
        String input = "/home/uranix/alerts.json";
        String output = "/home/uranix/new_compiled.json";

        ObjectMapper mapper = new ObjectMapper();
        try {
            BufferedReader scanner = new BufferedReader(new FileReader(input, StandardCharsets.UTF_8));
            List<String> programs = new ArrayList<>();
            while (true) {
                String line = scanner.readLine();
                if (line == null) {
                    break;
                }
                Code parsed = mapper.readValue(line, Code.class);
                programs.add(parsed.program);
            }

            PrintStream ps = new PrintStream(output);
            List<Object> compiled = compile(programs);
            for (var entry : compiled) {
                ps.println(mapper.writeValueAsString(entry.toString()));
            }

            return programs;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
