package ru.yandex.solomon.experiments.uranix;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.stockpile.api.MetricData;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class UncompressAlertingStatuses {

    @JsonInclude(JsonInclude.Include.NON_NULL)
    private static class Row {
        public String alertId;
        @Nullable
        public String parentId;
        public String projectId;

        public int shardId;
        public long localId;
    }

    private static Void messWithShardSafe(int shardId) {
        try {
            messWithShard(shardId);
            return null;
        } catch (Throwable t) {
            throw new RuntimeException(t);
        }
    }
    // 0 NODATA -> 3
    // 1 OK     -> 0
    // 2 XXX    -> x
    // 3 ERROR  -> 4
    // 4 ALARM  -> 2
    // 5 XXX    -> x
    // 6 WARN   -> 1
    // 7 HOLE   -> 5
    private static final int[] STATUS_TO_CODE = new int[] {3, 0, -1, 4, 2, -1, 1, 5};
    private static void messWithShard(int shardId) throws IOException {
        InputStream is = new GZIPInputStream(new FileInputStream("/home/uranix/alerting_statuses/" + shardId + ".pbzip"));
        GZIPOutputStream os = new GZIPOutputStream(new FileOutputStream("/home/uranix/alerting_statuses/" + shardId + ".zip"));
        MetricData metricData;

        long tick = System.nanoTime();
        while ((metricData = MetricData.parseDelimitedFrom(is)) != null) {
            Preconditions.checkState(metricData.getCompressed().getChunksCount() == 1);
            TimeSeries.Chunk chunk = metricData.getCompressed().getChunks(0);

            MetricArchiveImmutable
                archive = MetricArchiveNakedSerializer
                .serializerForFormatSealed(StockpileFormat.CURRENT).deserializeFull(chunk.getContent().toByteArray());

            var iterator = archive.iterator();
            AggrPoint point = new AggrPoint();

            ByteBuffer bb = ByteBuffer.allocate(8);
            bb.putLong(metricData.getLocalId());

            os.write(bb.array());

            long tsBegin = 1577456760000L;
            long ts = tsBegin;
            long deltaTs = 30000L;
            int inserted = 0;
            while (iterator.next(point)) {
                int val = STATUS_TO_CODE[(int)point.longValue];
                Preconditions.checkState(val >= 0);
                while (ts < point.getTsMillis()) {
                    //System.out.println("Inserted HOLE for " + shardId + ":" + metricData.localId() + " at " + ts);
                    os.write(STATUS_TO_CODE[7]);
                    ts += deltaTs;
                    inserted ++;
                }
                os.write(val);
                ts += deltaTs;
            }
            while (ts < tsBegin + 40320L * deltaTs) {
                //System.out.println("Inserted HOLE for " + shardId + ":" + metricData.localId() + " at END");
                os.write(STATUS_TO_CODE[7]);
                ts += deltaTs;
                inserted ++;
            }
            if (inserted > 0) {
                //System.out.println("Inserted " + inserted);
            }
        }
        os.finish();
        long elapsed = System.nanoTime() - tick;
        System.out.printf("Shard %d took %.3f ms\n", shardId, elapsed * 1e-6);
    }

    public static void main(String[] args) throws IOException {
        ObjectMapper mapper = new ObjectMapper();

        BufferedReader br = new BufferedReader(new FileReader("/home/uranix/alerting_statuses/alerting_statuses.meta"));
        HashMap<Integer, List<Row>> rowsByShardId = new HashMap<>();
        String s;
        while ((s = br.readLine()) != null) {
            Row row = mapper.readValue(s, Row.class);
            List<Row> shard = rowsByShardId.computeIfAbsent(row.shardId, ignore -> new ArrayList<>());
            shard.add(row);
        }

        List<Integer> shardIds = new ArrayList<>(rowsByShardId.keySet());

        ExecutorService executorService = Executors.newFixedThreadPool(50);

        List<CompletableFuture<Void>> futures = new ArrayList<>();

        for (var shardId : shardIds) {
            if (shardId == 3623) {
                futures.add(CompletableFutures.supplyAsync(() -> messWithShardSafe(shardId), executorService));
            }
        }

        CompletableFutures.allOfVoid(futures).join();
        executorService.shutdown();
    }
}
