package ru.yandex.solomon.experiments.gordiychuk.recovery;

import java.io.IOException;
import java.nio.file.Path;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.stockpile.server.shard.load.Async;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.experiments.gordiychuk.recovery.Records.randomRecord;

/**
 * @author Vladimir Gordiychuk
 */
public class AsyncMappingWriterTest {

    @Rule
    public TemporaryFolder tmp = new TemporaryFolder();

    private Path target;
    private AsyncMappingWriter writer;

    @Before
    public void setUp() throws IOException {
        target = tmp.newFile().toPath();
        writer = new AsyncMappingWriter(target);
    }

    @Test
    public void writeEmpty() {
        writer.complete();
        writer.doneFuture().join();

        try (var it = readIt()) {
            assertNull(it.next());
        }
    }

    @Test
    public void writeOne() {
        Record record = randomRecord();
        writer.write(record, record).join();
        writer.complete();
        writer.doneFuture().join();

        var expected = MappingRecord.of(record, record);
        try (var it = readIt()) {
            assertEquals(expected, it.next());
            assertNull(it.next());
        }
    }

    @Test
    public void writeMany() {
        int size = 10_000;
        List<Record> one = IntStream.range(0, size)
            .mapToObj(ignore -> randomRecord())
            .collect(Collectors.toList());
        List<Record> two = IntStream.range(0, size)
            .mapToObj(ignore -> randomRecord())
            .collect(Collectors.toList());

        {
            AtomicInteger index = new AtomicInteger();
            Async.forEach(() -> {
                var i = index.getAndIncrement();
                if (i >= size) {
                    return CompletableFuture.completedFuture(null);
                }

                return writer.write(one.get(i), two.get(i)).thenApply(ignore -> Boolean.TRUE);
            }, ignore -> {}).join();
            writer.complete();
            writer.doneFuture().join();
        }

        try (var it = readIt()) {
            for (int index = 0; index < size; index++) {
                var expected = MappingRecord.of(one.get(index), two.get(index));
                assertEquals(expected, it.next());
            }
            assertNull(it.next());
        }
    }

    @Test
    public void concurrentWriteMany() {
        List<Map.Entry<Record, Record>> source = IntStream.range(0, 10_000)
            .mapToObj(ignore -> Map.entry(randomRecord(), randomRecord()))
            .collect(Collectors.toList());

        {
            AtomicInteger index = new AtomicInteger();
            AsyncActorBody body = () -> {
                var i = index.getAndIncrement();
                if (i >= source.size()) {
                    return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
                }

                var entry = source.get(i);
                return CompletableFuture.supplyAsync(() -> writer.write(entry.getKey(), entry.getValue()));
            };

            new AsyncActorRunner(body, ForkJoinPool.commonPool(), 10).start().join();
        }
        writer.complete();
        writer.doneFuture().join();

        var expected = source.stream()
            .map(entry -> MappingRecord.of(entry.getKey(), entry.getValue()))
            .collect(Collectors.toSet());

        try (var it = readIt()) {
            while (!expected.isEmpty()) {
                var actual = it.next();
                assertTrue(String.valueOf(actual), expected.remove(actual));
            }
            assertNull(it.next());
        }
    }

    private MappingRecordIterator readIt() {
        return new MappingRecordIterator(target);
    }
}
