package ru.yandex.solomon.experiments.gordiychuk.recovery;

import java.io.IOException;
import java.nio.file.Path;
import java.util.List;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static ru.yandex.solomon.experiments.gordiychuk.recovery.Records.randomRecord;

/**
 * @author Vladimir Gordiychuk
 */
public class AsyncRecordWriterTest {
    @Rule
    public TemporaryFolder tmp = new TemporaryFolder();

    private Path target;
    private AsyncRecordWriter writer;

    @Before
    public void setUp() throws IOException {
        target = tmp.newFile().toPath();
        writer = new AsyncRecordWriter(target, ForkJoinPool.commonPool());
    }

    @Test
    public void writeEmpty() {
        writer.complete();
        writer.doneFuture().join();

        try (var it = readIt()) {
            assertNull(it.next());
        }
    }

    @Test
    public void writeOne() {
        Record record = randomRecord();
        writer.add(record);
        writer.complete();
        writer.doneFuture().join();

        try (var it = readIt()) {
            assertEquals(record, it.next());
            assertNull(it.next());
        }
    }

    @Test
    public void writeMany() {
        List<Record> records = IntStream.range(0, 10_000)
            .mapToObj(ignore -> randomRecord())
            .collect(Collectors.toList());

        for (var record : records) {
            writer.add(record);
        }
        writer.complete();
        writer.doneFuture().join();

        try (var it = readIt()) {
            for (var expected : records) {
                assertEquals(expected, it.next());
            }
            assertNull(it.next());
        }
    }

    @Test
    public void writeWithPause() throws InterruptedException {
        List<Record> partOne = IntStream.range(0, 10)
            .mapToObj(ignore -> randomRecord())
            .collect(Collectors.toList());

        for (var record : partOne) {
            writer.add(record);
        }

        TimeUnit.MILLISECONDS.sleep(10);

        List<Record> partTwo = IntStream.range(0, 100)
            .mapToObj(ignore -> randomRecord())
            .collect(Collectors.toList());

        for (var record : partTwo) {
            writer.add(record);
        }

        var done = writer.doneFuture();
        assertFalse(done.isDone());
        writer.complete();
        done.join();

        try (var it = readIt()) {
            for (var expected : partOne) {
                assertEquals(expected, it.next());
            }

            for (var expected : partTwo) {
                assertEquals(expected, it.next());
            }
            assertNull(it.next());
        }

    }

    private RecordIterator readIt() {
        return new RecordIterator(target);
    }
}
