package ru.yandex.solomon.experiments.gordiychuk.recovery;

import java.io.IOException;
import java.nio.file.Path;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.rules.TestName;

import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static ru.yandex.solomon.experiments.gordiychuk.recovery.Records.randomRecord;

/**
 * @author Vladimir Gordiychuk
 */
public class RecordIteratorTest {

    @Rule
    public TemporaryFolder tmp = new TemporaryFolder();
    @Rule
    public TestName testName = new TestName();

    private Path target;
    private RecordWriter writer;

    @Before
    public void setUp() throws IOException {
        String name = testName.getMethodName();
        target = tmp.newFile(name).toPath();
        writer = new RecordWriter(target);
    }

    @Test
    public void empty() {
        writer.close();

        RecordIterator it = new RecordIterator(target);
        assertNull(it.next());
    }

    @Test
    public void notExist() {
        var path = target.getParent().resolve("nonExist");
        RecordIterator it = new RecordIterator(path);
        assertNull(it.next());
    }

    @Test
    public void readOne() {
        Record record = new Record();
        record.labels = "&name=alice&tag=test&";
        record.shardId = StockpileShardId.random();
        record.localId = StockpileLocalId.random();
        record.flags = 5;

        writer.write(record);
        writer.close();

        RecordIterator it = new RecordIterator(target);
        var read = it.next();
        assertNotNull(read);
        assertEquals(record, read);

        assertNull(it.next());
        assertNull(it.next());
        assertNull(it.next());
        it.close();
    }

    @Test
    public void readMany() {
        List<Record> records = IntStream.range(0, 1000)
            .mapToObj(index -> randomRecord())
            .peek(writer::write)
            .collect(Collectors.toList());
        writer.close();

        RecordIterator it = new RecordIterator(target);

        for (Record expected : records) {
            var actual = it.next();
            assertEquals(expected, actual);
        }

        assertNull(it.next());
        assertNull(it.next());
        it.close();
    }
}
