package ru.yandex.solomon.experiments.gordiychuk.recovery;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.IntStream;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.rules.TestName;

import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.kikimrKv.ShardIdMapToLong;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.experiments.gordiychuk.recovery.Records.randomMappingRecord;

/**
 * @author Vladimir Gordiychuk
 */
public class UploadMappingToShardsTest {

    @Rule
    public TemporaryFolder tmp = new TemporaryFolder();
    @Rule
    public TestName testName = new TestName();

    private final String volumePath = "/Solomon/Solomon/Kv";
    private Path target;
    private KikimrKvClientInMem kvClient = new KikimrKvClientInMem();
    private ShardIdMapToLong mapping;

    @Before
    public void setUp() throws IOException {
        String name = testName.getMethodName();
        target = tmp.newFolder(name).toPath();
        Files.createDirectories(target.resolve("mapping"));
        kvClient.createKvTablets(volumePath, 50).join();
        mapping = kvClient.resolveKvTablets(volumePath).thenApply(ShardIdMapToLong::new).join();
    }

    @Test
    public void none() {
        runTask();
        var files = mapping.shardIdStream()
            .mapToObj(shardId -> {
                return kvClient.readRangeNames(mapping.get(shardId), 0, NameRange.single("mapping"), expiredAt());
            })
            .collect(collectingAndThen(toList(), CompletableFutures::allOf))
            .join()
            .stream()
            .flatMap(Collection::stream)
            .collect(toList());

        assertEquals(0, files.size());
    }

    @Test
    public void oneShard() {
        write(42, IntStream.range(0, 1_000_000)
            .mapToObj(ignore -> randomMappingRecord())
            .collect(toList()));

        runTask();
        assertEquals(readFromDisk(42), readFromKv(42));
        assertEquals("", readFromKv(43));
    }

    @Test
    public void manyShards() {
        mapping.shardIdStream()
            .parallel()
            .forEach(shardId -> {
                var records = IntStream.range(0, ThreadLocalRandom.current().nextInt(1, 1000))
                    .mapToObj(ignore -> randomMappingRecord())
                    .collect(toList());
                write(shardId, records);
            });

        runTask();

        mapping.shardIdStream()
            .forEach(shardId -> {
                var disk = readFromDisk(shardId);
                var kv = readFromKv(shardId);
                assertEquals("shard: " + shardId, disk, kv);
            });
    }

    private void write(int shardId, MappingRecord record) {
        write(shardId, List.of(record));
    }

    private void write(int shardId, List<? extends MappingRecord> records) {
        try(var writer = new MappingWriter(target.resolve("mapping").resolve(StockpileShardId.toString(shardId)))) {
            writer.writeAll(records);
        }
    }

    private String readFromKv(int shardId) {
        var tabletId = mapping.get(shardId);
        var files = kvClient.readRangeNames(tabletId, 0, NameRange.single("mapping"), expiredAt()).join();
        if (files.isEmpty()) {
            return "";
        }

        var bytes = kvClient.readDataLarge(tabletId, 0, "mapping", expiredAt(), MsgbusKv.TKeyValueRequest.EPriority.REALTIME).join();
        return new String(bytes);
    }

    private String readFromDisk(int shardId) {
        try {
            return Files.readString(target.resolve("mapping").resolve("" + shardId));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static long expiredAt() {
        return System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(1);
    }

    private void runTask() {
        new UploadMappingToShards(target, volumePath, kvClient).run();
    }
}
