package ru.yandex.solomon.experiments.gordiychuk.recovery.metabase;

import java.io.IOException;
import java.nio.file.Path;
import java.util.List;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.rules.TestName;

import ru.yandex.devtools.test.annotations.YaIgnore;
import ru.yandex.solomon.experiments.gordiychuk.recovery.Record;
import ru.yandex.solomon.experiments.gordiychuk.recovery.RecordIterator;
import ru.yandex.solomon.kikimr.LocalKikimr;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static ru.yandex.solomon.experiments.gordiychuk.recovery.Records.randomRecord;

/**
 * @author Vladimir Gordiychuk
 */
@YaIgnore
public class DumpMetricsFromShardTaskTest {
    @ClassRule
    public static LocalKikimr localKikimr = new LocalKikimr();

    @Rule
    public TemporaryFolder folder = new TemporaryFolder();

    @Rule
    public TestName testName = new TestName();

    private YdbClient ydb;
    private MetabaseShardMetricsReader reader;
    private String root;
    private DumpMetricsFromShardTask task;
    private Path target;

    @Before
    public void setUp() throws IOException {
        root = localKikimr.getRoot().resolve(testName.getMethodName()).toString();
        target = folder.newFile().toPath();

        ydb = YdbHelper.createYdbClient(localKikimr.getGrpcEndpoint());
        reader = new MetabaseShardMetricsReader(root, ydb, "alice", 0);
        task = new DumpMetricsFromShardTask(root, ydb, "alice", ForkJoinPool.commonPool(), target);
        MetabaseUtils.createSchema(ydb, root, "alice");
    }

    @Test
    public void emptyTable() {
        task.run().join();

        try (var it = new RecordIterator(target)) {
            assertNull(it.next());
        }
    }

    @Test
    public void oneRecord() {
        Record record = randomRecord();
        record.createdAtSeconds = nowSeconds();
        MetabaseUtils.insert(ydb, root, "alice", record);

        task.run().join();

        try (var it = new RecordIterator(target)) {
            assertEquals(record, it.next());
            assertNull(it.next());
        }
    }

    @Test
    public void many() {
        List<Record> records = IntStream.range(0, 100_000)
            .mapToObj(index -> randomRecord())
            .sorted(Record::compareTo)
            .collect(Collectors.toList());
        MetabaseUtils.insert(ydb, root, "alice", records);

        task.run().join();

        try(var it = new RecordIterator(target)) {
            for (Record record : records) {
                assertEquals(record, it.next());
            }
            assertNull(it.next());
        }
    }

    private long nowSeconds() {
        return TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()) - TimeUnit.DAYS.toSeconds(10);
    }
}
