package ru.yandex.solomon.experiments.gordiychuk.recovery.metabase;

import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.rules.TestName;

import ru.yandex.solomon.experiments.gordiychuk.recovery.MappingRecord;
import ru.yandex.solomon.experiments.gordiychuk.recovery.MappingRecordIterator;
import ru.yandex.solomon.experiments.gordiychuk.recovery.Record;
import ru.yandex.solomon.experiments.gordiychuk.recovery.RecordIterator;
import ru.yandex.solomon.experiments.gordiychuk.recovery.RecordWriter;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static ru.yandex.solomon.experiments.gordiychuk.recovery.Records.randomRecord;

/**
 * @author Vladimir Gordiychuk
 */
public class MetabaseFileMetricsMergerTest {

    @Rule
    public TemporaryFolder tmp = new TemporaryFolder();
    @Rule
    public TestName testName = new TestName();

    private Path root;
    private Unit alice;
    private Unit bob;
    @Before
    public void setUp() throws IOException {
        root = tmp.newFolder().toPath();
        alice = new Unit("alice");
        bob = new Unit("bob");
    }

    @Test
    public void mergeSame() {
        Record one = new Record();
        one.labels = "&name=grpc.client.resource.usage&";
        one.shardId = StockpileShardId.random();
        one.localId = StockpileLocalId.random();
        alice.write("test", one);

        Record two = new Record();
        two.labels = "&name=grpc.client.resource.usage&";
        two.shardId = StockpileShardId.random();
        two.localId = StockpileLocalId.random();
        bob.write("test", two);

        MetabaseFileMetricsMerger.merge(alice.source, bob.source);
        try (var it = alice.nonExists("test")) {
            assertNull(it.next());
        }
        try (var it = bob.nonExists("test")) {
            assertNull(it.next());
        }

        try (var it = alice.mapping(one.shardId)) {
            var next = it.next();
            assertNotNull(next);
            assertEquals(one.localId, next.localLocalId);
            assertEquals(two.shardId, next.remoteShardId);
            assertEquals(two.localId, next.remoteLocalId);
            assertNull(it.next());
        }

        try (var it = bob.mapping(two.shardId)) {
            var next = it.next();
            assertNotNull(next);
            assertEquals(two.localId, next.localLocalId);
            assertEquals(one.shardId, next.remoteShardId);
            assertEquals(one.localId, next.remoteLocalId);
            assertNull(it.next());
        }
    }

    @Ignore
    public void mergeDifferent() {
        List<Record> one = new ArrayList<>();
        one.add(record("&name=alice&"));
        one.add(record("&name=eva&"));
        for (var record : one) {
            alice.write(testName.getMethodName(), record);
        }

        List<Record> two = new ArrayList<>();
        two.add(record("&name=bob&"));
        two.add(record("&name=eva&"));
        two.add(record("&name=bob2&")); // hash code order
        for (var record : two) {
            bob.write(testName.getMethodName(), record);
        }

        MetabaseFileMetricsMerger.merge(alice.source, bob.source);
        try (var it = alice.nonExists(testName.getMethodName())) {
            assertEquals(two.get(0), it.next());
            assertEquals(two.get(2), it.next());
            assertNull(it.next());
        }
        try (var it = bob.nonExists(testName.getMethodName())) {
            assertEquals(one.get(0), it.next());
            assertNull(it.next());
        }


        alice.assetMappingEquals(
            List.of(
                Map.entry(one.get(0), one.get(0)),
                Map.entry(one.get(1), two.get(1))
            )
        );

        bob.assetMappingEquals(
            List.of(
                Map.entry(two.get(0), two.get(0)),
                Map.entry(two.get(1), one.get(1)),
                Map.entry(two.get(2), two.get(2))
            )
        );
    }

    @Test
    public void ignoreShardAbsentOnAnother() {
        List<Record> one = new ArrayList<>();
        one.add(record("&name=alice&"));
        one.add(record("&name=eva&"));
        for (var record : one) {
            alice.write(testName.getMethodName(), record);
        }

        List<Record> two = new ArrayList<>();
        two.add(record("&name=bob&"));
        two.add(record("&name=eva&"));
        two.add(record("&name=bob2&")); // hash code order
        for (var record : two) {
            bob.write("absent", record);
        }

        MetabaseFileMetricsMerger.merge(alice.source, bob.source);
        try (var it = alice.nonExists(testName.getMethodName())) {
            assertNull(it.next());
        }
        try (var it = bob.nonExists(testName.getMethodName())) {
            assertNull(it.next());
        }

        alice.assetMappingEquals(List.of());
        for (Record r : one) {
            try (var it = alice.mapping(r.shardId)) {
                assertNull(it.next());
            }
        }
    }

    @Test
    public void mergeMultipleShards() {
        for (int shardId = 0; shardId < 10; shardId++) {
            for (int index = 0; index < 1000; index++) {
                String shardStr = String.format("%09d", index++);
                var record = randomRecord();
                record.shardId = ThreadLocalRandom.current().nextInt(42, 45);
                alice.write(shardStr, record);
                if (ThreadLocalRandom.current().nextDouble() >= 0.2) {
                    bob.write(shardStr, record);
                }
            }
        }

        MetabaseFileMetricsMerger.merge(alice.source, bob.source);
    }

    public Record record(String labels) {
        var record = randomRecord();
        record.labels = labels;
        return record;
    }

    public MappingRecord map(Record local, Record remote) {
        MappingRecord record = new MappingRecord();
        record.localLocalId = local.localId;
        record.remoteShardId = remote.shardId;
        record.remoteLocalId = remote.localId;
        return record;
    }

    private class Unit {
        String name;
        Path source;

        public Unit(String name) throws IOException {
            this.name = name;
            this.source = root.resolve(name);
        }

        public void write(String shardId, Record record) {
            Path file = source.resolve("metrics").resolve(shardId);
            try (var writer = new RecordWriter(file)) {
                writer.write(record);
            }
            System.out.println(String.format("%15s", file + "#write ") + record);
        }

        public RecordIterator nonExists(String shardId) {
            return new RecordIterator(source.resolve("absent").resolve(shardId));
        }

        public MappingRecordIterator mapping(int shardId) {
            return new MappingRecordIterator(source.resolve("mapping").resolve("" + shardId));
        }

        private void assetMappingEquals(List<Map.Entry<Record, Record>> entries) {
            var sorted = entries.stream()
                .sorted(Comparator.comparingInt(o -> o.getKey().shardId))
                .collect(Collectors.toList());

            for (int index = 0; index < sorted.size(); index++) {
                var entry = entries.get(index);
                var local = entry.getKey();
                var remote = entry.getValue();
                var expected = map(local, remote);
                try (var it = mapping(local.shardId)) {
                    do {
                        var actual = it.next();
                        assertEquals(local.toString(), expected, actual);
                        if (index + 1 >= sorted.size()) {
                            return;
                        }

                        if (sorted.get(index + 1).getKey().shardId == local.shardId) {
                            index++;
                            entry = entries.get(index);
                            local = entry.getKey();
                            remote = entry.getValue();
                            expected = map(local, remote);
                            assertEquals(local.toString(), expected, actual);
                        } else {
                            assertNull(it.next());
                            break;
                        }
                    } while (true);
                }
            }
        }
    }
}
