package ru.yandex.solomon.experiments.gordiychuk.recovery.metabase;

import java.util.List;
import java.util.Map;

import com.google.common.collect.Lists;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;

import ru.yandex.solomon.experiments.gordiychuk.recovery.Record;
import ru.yandex.solomon.tool.YdbClient;

/**
 * @author Vladimir Gordiychuk
 */
public class MetabaseUtils {
    public static void createSchema(YdbClient ydb, String root, String shardId) {
        String path = root + "/Solomon/metrics";
        ydb.scheme.makeDirectories(path).join().expect("cannot create dir " + path);
        ydb.fluent()
            .retryForever()
            .createTable(root + "/Solomon/metrics/"+shardId, TableDescription.newBuilder()
                .addNullableColumn("hash", PrimitiveType.uint32())
                .addNullableColumn("labels", PrimitiveType.utf8())
                .addNullableColumn("labels", PrimitiveType.utf8())
                .addNullableColumn("shardId", PrimitiveType.uint32())
                .addNullableColumn("localId", PrimitiveType.uint64())
                .addNullableColumn("createdSeconds", PrimitiveType.uint64())
                .addNullableColumn("flags", PrimitiveType.uint32())
                .setPrimaryKeys("hash", "labels")
                .build())
            .join()
            .expect("created");
        System.out.println("Created table "+ root + "/Solomon/metrics/" + shardId);
    }

    public static void insert(YdbClient ydb, String root, String shardId, Record record) {
        insert(ydb, root, shardId, List.of(record));
    }

    public static void insert(YdbClient ydb, String root, String shardId, List<Record> records) {
        var chunks = Lists.partition(records, 50_000);
        for (var chunk : chunks) {
            StructValue[] structs = chunk.stream().map(MetabaseUtils::toStruct).toArray(StructValue[]::new);
            ListValue listStructs = ListType.of(structs[0].getType()).newValueOwn(structs);
            Params params = Params.of("$input", listStructs);

            String query = String.format("""
                    --!syntax_v1
                    DECLARE $input AS List<Struct<hash:Uint32,labels:Utf8,shardId:Uint32,localId:Uint64,createdSeconds:Uint64,flags:Uint32>>;
                    UPSERT INTO `%s/Solomon/metrics/%s`
                    SELECT * FROM AS_TABLE($input);
                    """, root, shardId);

            ydb.fluent()
                .retryForever()
                .execute(query, params)
                .thenApply(Result::toStatus)
                .join()
                .expect("inserted");
            System.out.println("Insert into "+ root + "/Solomon/metrics/" + shardId + " " + chunk.size() + " records");
        }
    }

    private static StructValue toStruct(Record record) {
        return StructValue.of(Map.of(
            "hash", PrimitiveValue.uint32(record.labels.hashCode()),
            "labels", PrimitiveValue.utf8(record.labels),
            "shardId", PrimitiveValue.uint32(record.shardId),
            "localId", PrimitiveValue.uint64(record.localId),
            "createdSeconds", PrimitiveValue.uint64(record.createdAtSeconds),
            "flags", PrimitiveValue.uint32(record.flags)
        ));
    }

}
