package ru.yandex.solomon.experiments.gordiychuk.recovery.metabase;

import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.rules.TestName;

import ru.yandex.devtools.test.annotations.YaIgnore;
import ru.yandex.solomon.experiments.gordiychuk.recovery.Record;
import ru.yandex.solomon.experiments.gordiychuk.recovery.RecordWriter;
import ru.yandex.solomon.kikimr.LocalKikimr;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.experiments.gordiychuk.recovery.Records.randomRecord;

/**
 * @author Vladimir Gordiychuk
 */
@YaIgnore
public class RestoreNonExistsMetricsTaskTest {
    @ClassRule
    public static LocalKikimr localKikimr = new LocalKikimr();

    @Rule
    public TemporaryFolder tmp = new TemporaryFolder();
    @Rule
    public TestName testName = new TestName();

    private String root;
    private YdbClient ydb;
    private Path file;

    @Before
    public void setUp() throws IOException {
        root = localKikimr.getRoot().resolve(testName.getMethodName()).toString();
        ydb = YdbHelper.createYdbClient(localKikimr.getGrpcEndpoint());
        file = tmp.newFolder().toPath();
    }

    @After
    public void tearDown() {
        ydb.close();
    }

    @Test
    public void onlyOneShard() {
        MetabaseUtils.createSchema(ydb, root, "alice");
        List<Record> expected = IntStream.range(0, 150_000)
            .mapToObj(ignore -> randomRecord())
            .sorted(Record::compareTo)
            .collect(Collectors.toList());

        try (var writer = new RecordWriter(file.resolve("absent").resolve("alice"))) {
            expected.forEach(writer::write);
        }

        run();
        var result = read("alice");
        assertArrayEquals(expected.toArray(), result.toArray());
    }

    @Test
    public void fewShard() {
        MetabaseUtils.createSchema(ydb, root, "alice");
        List<Record> alice = IntStream.range(0, ThreadLocalRandom.current().nextInt(1, 10000))
            .mapToObj(ignore -> randomRecord())
            .sorted(Record::compareTo)
            .collect(Collectors.toList());
        try (var writer = new RecordWriter(file.resolve("absent").resolve("alice"))) {
            alice.forEach(writer::write);
        }

        MetabaseUtils.createSchema(ydb, root, "bob");
        List<Record> bob = IntStream.range(0, ThreadLocalRandom.current().nextInt(1, 1000))
            .mapToObj(ignore -> randomRecord())
            .sorted(Record::compareTo)
            .collect(Collectors.toList());
        try (var writer = new RecordWriter(file.resolve("absent").resolve("bob"))) {
            bob.forEach(writer::write);
        }

        run();

        assertArrayEquals(alice.toArray(), read("alice").toArray());
        assertArrayEquals(bob.toArray(), read("bob").toArray());
    }

    @Test
    public void upsertMetric() {
        MetabaseUtils.createSchema(ydb, root, "alice");
        var source = randomRecord();
        MetabaseUtils.insert(ydb, root, "alice", source);

        var update = randomRecord();
        update.labels = source.labels;

        try (var writer = new RecordWriter(file.resolve("absent").resolve("alice"))) {
            writer.write(update);
        }

        run();

        var read = read("alice");
        assertEquals(List.of(update), read);
    }

    public void run() {
        new RestoreNonExistsMetrics(root, ydb, file, ForkJoinPool.commonPool()).run().join();
    }

    public List<Record> read(String shardId) {
        var reader = new MetabaseShardMetricsReader(root, ydb, shardId, 0);
        List<Record> records = new ArrayList<>();
        reader.run(records::add).join();
        return records;
    }
}
