package ru.yandex.solomon.experiments.gordiychuk.recovery.stockpile;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicInteger;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.experiments.gordiychuk.recovery.MappingRecord;
import ru.yandex.solomon.experiments.gordiychuk.recovery.MappingWriter;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.TsRandomData;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.ChunkWithNo;
import ru.yandex.stockpile.server.data.chunk.ChunkWriter;
import ru.yandex.stockpile.server.data.index.LongFileTypeSnapshotIndex;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.shard.SnapshotReason;
import ru.yandex.stockpile.server.shard.load.Async;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static ru.yandex.solomon.experiments.gordiychuk.recovery.Records.randomMappingRecord;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileShardReaderTest {

    private ChunkWriter writer;
    private KikimrKvClientInMem kvClient;
    private long txn = 42;
    private long tabletId;
    private StockpileShardReader reader;

    @Rule
    public TemporaryFolder tmp = new TemporaryFolder();

    @Before
    public void setUp() {
        writer = new ChunkWriter(SnapshotReason.SPACE, System.currentTimeMillis(), 0);
        kvClient = new KikimrKvClientInMem();
        tabletId = kvClient.createKvTablet();
        reader = new StockpileShardReader(kvClient);
    }

    @Test
    public void makeBackup() {
        int mask = TsColumn.mask | ValueColumn.mask;
        long localId = StockpileLocalId.random();
        for (int index = 0; index < 2_000_000; index++) {
            var archive = randomArchive(mask, ThreadLocalRandom.current().nextInt(1, 2));
            localId += ThreadLocalRandom.current().nextInt(1, 10000);
            write(localId, archive);
        }
        complete();

        reader.makeBackup(tabletId).join();

        kvClient.deleteRanges(tabletId, 0, StockpileKvNames.currentSnapshot(SnapshotLevel.ETERNITY), 0).join();

        var files = kvClient.readRangeNames(tabletId, 0, 0).join();
        assertNotEquals(0, files.size());
    }

    @Test
    public void deleteBackup() {
        int mask = TsColumn.mask | ValueColumn.mask;
        long localId = StockpileLocalId.random();
        for (int index = 0; index < 2_000_000; index++) {
            var archive = randomArchive(mask, ThreadLocalRandom.current().nextInt(1, 2));
            localId += ThreadLocalRandom.current().nextInt(1, 10000);
            write(localId, archive);
        }
        complete();

        reader.makeBackup(tabletId).join();

        var prevFiles = kvClient.readRangeNames(tabletId, 0, 0).join();

        var currentFiles = kvClient.readRangeNames(tabletId, 0, FileNamePrefix.Current.instance.toNameRange(), 0).join();
        reader.dropBackup(tabletId).join();

        var nowFiles = kvClient.readRangeNames(tabletId, 0, 0).join();
        assertEquals(prevFiles.size() / 2, nowFiles.size());
        assertEquals(currentFiles, nowFiles);
    }

    @Test
    public void iterate() {
        int mask = TsColumn.mask | ValueColumn.mask;
        long localId = StockpileLocalId.random();
        List<Map.Entry<Long, MetricArchiveImmutable>> expected = new ArrayList<>(1_000_000);
        for (int index = 0; index < 1_000_000; index++) {
            var archive = randomArchive(mask, ThreadLocalRandom.current().nextInt(1, 2));
            localId += ThreadLocalRandom.current().nextInt(1, 10000);
            write(localId, archive);
            expected.add(Map.entry(localId, archive));
        }
        complete();

        reader.makeBackup(tabletId).join();

        // iterate by snapshots
        kvClient.deleteRanges(tabletId, 0, StockpileKvNames.currentSnapshot(SnapshotLevel.ETERNITY), 0).join();

        AtomicInteger index = new AtomicInteger();
        Async.forEach(reader.metricsIterator(tabletId).join(), data -> {
            var i = index.getAndIncrement();
            var entry = expected.get(i);
            assertEquals(entry.getKey().longValue(), data.localId());
            assertEquals(entry.getValue(), data.archive());
        }).join();
        assertEquals(index.get(), expected.size());

        reader.dropBackup(tabletId).join();
    }

    @Test
    public void readEmptyMapping() {
        var result = reader.readMapping(tabletId).join();
        assertEquals(List.of(), result);
    }

    @Test
    public void readMappingSorted() throws IOException {
        Path file = tmp.newFile().toPath();
        List<MappingRecord> expected = new ArrayList<>(10_000_000);
        try(var w = new MappingWriter(file)) {
            var record = randomMappingRecord();
            w.write(record);
            expected.add(record);
        }
        expected.sort((o1, o2) -> Long.compareUnsigned(o1.localLocalId, o2.localLocalId));

        kvClient.write(
            tabletId,
            0,
            "mapping",
            Files.readAllBytes(file),
            MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN,
            MsgbusKv.TKeyValueRequest.EPriority.REALTIME,
            0).join();

        var result = reader.readMapping(tabletId).join();
        assertEquals(expected, result);

        reader.deleteMapping(tabletId).join();
        assertEquals(List.of(), reader.readMapping(tabletId).join());
    }

    private MetricArchiveImmutable randomArchive(int mask, int points) {
        try (MetricArchiveMutable archive = new MetricArchiveMutable()) {
            RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
            long tsMillis = TsRandomData.randomTs(ThreadLocalRandom.current());
            for (int index = 0; index < points; index++) {
                tsMillis += 10_000;
                randomPoint(point, mask);
                point.tsMillis = tsMillis;
                archive.addRecordData(mask, point);
            }
            return archive.toImmutableNoCopy();
        }
    }

    private void write(long localId, MetricArchiveImmutable archive) {
        var chunk = writer.writeMetric(localId, System.currentTimeMillis(), archive);
        if (chunk != null) {
            writeChunk(chunk);
        }
    }

    private void complete() {
        var result = writer.finish();
        if (result.chunkWithNo != null) {
            writeChunk(result.chunkWithNo);
        }

        var writes = LongFileTypeSnapshotIndex.I.makeWrites(
            LongFileTypeSnapshotIndex.I.serialize(new SnapshotIndex(SnapshotLevel.ETERNITY, txn, result.index)),
            MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN,
            MsgbusKv.TKeyValueRequest.EPriority.REALTIME);

        kvClient.writeAndRename(tabletId, 0, writes, List.of(), 0).join();
    }

    private void writeChunk(ChunkWithNo chunk) {
        String chunkFileName = StockpileKvNames.chunkFileName(SnapshotLevel.ETERNITY, txn, chunk.getNo(), FileNamePrefix.Current.instance);
        kvClient.write(tabletId, 0, chunkFileName, chunk.getContent(), MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN, MsgbusKv.TKeyValueRequest.EPriority.REALTIME, 0).join();
    }
}
