package ru.yandex.solomon.experiments.gordiychuk.recovery.stockpile;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;

import it.unimi.dsi.fastutil.longs.Long2ObjectLinkedOpenHashMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.experiments.gordiychuk.recovery.MappingRecord;
import ru.yandex.solomon.experiments.gordiychuk.recovery.MappingWriter;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.TsRandomData;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.ChunkWithNo;
import ru.yandex.stockpile.server.data.chunk.ChunkWriter;
import ru.yandex.stockpile.server.data.index.LongFileTypeSnapshotIndex;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.shard.SnapshotReason;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.experiments.gordiychuk.recovery.Records.randomMappingRecord;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileShardRecoveryTest {

    private ChunkWriter writer;
    private KikimrKvClientInMem kvClient;
    private long txn = ThreadLocalRandom.current().nextInt(1, 1000);
    private long tabletId;
    private StockpileShardReader reader;
    private StockpileClientStub stockpile;
    private StockpileShardRecovery recovery;

    @Rule
    public TemporaryFolder tmp = new TemporaryFolder();

    @Before
    public void setUp() {
        writer = new ChunkWriter(SnapshotReason.SPACE, System.currentTimeMillis(), 0);
        kvClient = new KikimrKvClientInMem();
        tabletId = kvClient.createKvTablet();
        reader = new StockpileShardReader(kvClient);
        stockpile = new StockpileClientStub(ForkJoinPool.commonPool());
        recovery = new StockpileShardRecovery(reader, stockpile, ForkJoinPool.commonPool());
    }

    @Test
    public void skipRecoveryNoMapping() {
        int mask = TsColumn.mask | ValueColumn.mask;
        long localId = StockpileLocalId.random();
        for (int index = 0; index < 5; index++) {
            var archive = randomArchive(mask, ThreadLocalRandom.current().nextInt(1, 2));
            localId += ThreadLocalRandom.current().nextInt(1, 10000);
            write(localId, archive);
        }
        complete();

        recovery.run(tabletId).join();
        assertEquals(0, stockpile.getTimeSeriesCount());
    }

    @Test
    public void recoverOnlyMappedMetrics() {
        ThreadLocalRandom random = ThreadLocalRandom.current();
        List<MappingRecord> mapping = new ArrayList<>();
        Long2ObjectMap<MetricArchiveImmutable> expected = new Long2ObjectLinkedOpenHashMap<>();
        int mask = TsColumn.mask | ValueColumn.mask;
        long localId = StockpileLocalId.random();
        for (int index = 0; index < 100_000; index++) {
            var archive = randomArchive(mask, random.nextInt(1, 100));
            localId += random.nextInt(1, 10000);
            write(localId, archive);
            expected.put(localId, archive);
            if (random.nextDouble() > 0.6) {
                var map = randomMappingRecord();
                map.localLocalId = localId;
                mapping.add(map);
            }
        }
        complete();

        Collections.shuffle(mapping, random);
        writeMapping(mapping);

        recovery.run(tabletId).join();
        assertEquals(mapping.size(), stockpile.getTimeSeriesCount());
        for (var m : mapping) {
            var expect = expected.get(m.localLocalId);
            var actual = stockpile.getTimeSeries(m.remoteShardId, m.remoteLocalId);
            assertEquals(expect, actual);
        }

        assertEquals(List.of(), reader.readMapping(tabletId).join());
    }

    private MetricArchiveImmutable randomArchive(int mask, int points) {
        try(var archive = new MetricArchiveMutable()) {
            RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
            long tsMillis = TsRandomData.randomTs(ThreadLocalRandom.current());
            for (int index = 0; index < points; index++) {
                tsMillis += 10_000;
                randomPoint(point, mask);
                point.tsMillis = tsMillis;
                archive.addRecordData(mask, point);
            }
            return archive.toImmutableNoCopy();
        }
    }

    private void writeMapping(List<MappingRecord> records) {
        try {
            Path file = tmp.newFile().toPath();

            try(var w = new MappingWriter(file)) {
                records.forEach(w::write);
            }

            kvClient.write(
                tabletId,
                0,
                "mapping",
                Files.readAllBytes(file),
                MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN,
                MsgbusKv.TKeyValueRequest.EPriority.REALTIME,
                0).join();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void write(long localId, MetricArchiveImmutable archive) {
        var chunk = writer.writeMetric(localId, System.currentTimeMillis(), archive);
        if (chunk != null) {
            writeChunk(chunk);
        }
    }

    private void complete() {
        var result = writer.finish();
        if (result.chunkWithNo != null) {
            writeChunk(result.chunkWithNo);
        }

        var writes = LongFileTypeSnapshotIndex.I.makeWrites(
            LongFileTypeSnapshotIndex.I.serialize(new SnapshotIndex(SnapshotLevel.ETERNITY, txn, result.index)),
            MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN,
            MsgbusKv.TKeyValueRequest.EPriority.REALTIME);

        kvClient.writeAndRename(tabletId, 0, writes, List.of(), 0).join();
    }

    private void writeChunk(ChunkWithNo chunk) {
        String chunkFileName = StockpileKvNames.chunkFileName(SnapshotLevel.ETERNITY, txn, chunk.getNo(), FileNamePrefix.Current.instance);
        kvClient.write(tabletId, 0, chunkFileName, chunk.getContent(), MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN, MsgbusKv.TKeyValueRequest.EPriority.REALTIME, 0).join();
    }
}
