package ru.yandex.solomon.experiments.uranix;

import java.io.IOException;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.logging.log4j.Level;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.api.converters.MuteConverter;
import ru.yandex.solomon.alert.charts.ChartsClient;
import ru.yandex.solomon.alert.charts.ChartsClientImpl;
import ru.yandex.solomon.alert.cluster.broker.mute.ProjectMuteService;
import ru.yandex.solomon.alert.cluster.broker.mute.search.MuteSearch;
import ru.yandex.solomon.alert.dao.TelegramDao;
import ru.yandex.solomon.alert.dao.TelegramEventsDao;
import ru.yandex.solomon.alert.dao.memory.InMemoryMutesDao;
import ru.yandex.solomon.alert.dao.memory.InMemoryTelegramDao;
import ru.yandex.solomon.alert.dao.memory.InMemoryTelegramEventsDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.domain.expression.ExpressionAlert;
import ru.yandex.solomon.alert.notification.TemplateVarsFactory;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationChannel;
import ru.yandex.solomon.alert.notification.channel.telegram.ChatIdStorage;
import ru.yandex.solomon.alert.notification.channel.telegram.MuteApiStub;
import ru.yandex.solomon.alert.notification.channel.telegram.RawTelegramLimits;
import ru.yandex.solomon.alert.notification.channel.telegram.TelegramNotificationChannelFactory;
import ru.yandex.solomon.alert.notification.channel.telegram.TelegramTemplate;
import ru.yandex.solomon.alert.notification.channel.telegram.TelegramUpdatesReceiver;
import ru.yandex.solomon.alert.notification.domain.telegram.TelegramNotification;
import ru.yandex.solomon.alert.rule.AlertMuteStatus;
import ru.yandex.solomon.alert.rule.AlertProcessingState;
import ru.yandex.solomon.alert.rule.EvaluationState;
import ru.yandex.solomon.alert.telegram.DefaultTelegramClient;
import ru.yandex.solomon.alert.telegram.TelegramClient;
import ru.yandex.solomon.alert.template.MustacheTemplateFactory;
import ru.yandex.solomon.config.protobuf.Time;
import ru.yandex.solomon.config.protobuf.alert.TelegramChannelConfig;
import ru.yandex.solomon.locks.DistributedLockStub;
import ru.yandex.solomon.main.logger.LoggerConfigurationUtils;
import ru.yandex.solomon.util.host.HostUtils;
import ru.yandex.staff.StaffClient;
import ru.yandex.staff.StaffClientOptions;
import ru.yandex.staff.StaffClients;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class TelegramBot {
    private static final Logger logger = LoggerFactory.getLogger(TelegramBot.class);

    private final ScheduledExecutorService timer;
    private final NotificationChannel channel;

    private final ActorWithFutureRunner actor;
    private final Alert alert;
    private final Alert multiAlert;
    private final SubAlert subAlert1;
    private final SubAlert subAlert2;

    public TelegramBot(String[] args) throws IOException, InterruptedException {
        LoggerConfigurationUtils.simpleLogger(Level.DEBUG);

        String token = args[0];
        String botToken = args[1];

        timer = Executors.newSingleThreadScheduledExecutor();
        ExecutorService executor = Executors.newFixedThreadPool(2);

        TelegramClient telegramClient = new DefaultTelegramClient(
                "https://api.telegram.org",
                botToken,
                executor
        );

        TelegramChannelConfig config = TelegramChannelConfig.newBuilder()
                .setBotName("UranixStickerBot")
                .setFetchPeriod(Time.newBuilder()
                        .setValue(5)
                        .setUnit(ru.yandex.solomon.config.protobuf.TimeUnit.SECONDS)
                        .build())
                .build();
        TelegramDao dao = new InMemoryTelegramDao();
        StaffClient staffClient = StaffClients.create(StaffClientOptions.newBuilder()
                .setExecutor(executor)
                .setCacheTtl(5, TimeUnit.MINUTES)
                .setOauthToken(token)
                .build());
        DistributedLockStub distributedLock = new DistributedLockStub(Clock.systemUTC());
        distributedLock.setOwner(HostUtils.getFqdn());
        ChatIdStorage resolver = new ChatIdStorageStub();
        TelegramEventsDao telegramEventsDao = new InMemoryTelegramEventsDao();

        ChartsClient chartsClient = new ChartsClientImpl(executor, token, "production", Duration.ofSeconds(30));
        //ChartsClient chartsClient = new ChartsClientImpl(executor, token, "prestable", Duration.ofSeconds(60));

        var mutesDao = new InMemoryMutesDao();
        var mutesSearch = new MuteSearch(MuteConverter.INSTANCE);
        var projectMuteService = new ProjectMuteService(
                "junk",
                Clock.systemUTC(),
                mutesDao,
                mutesSearch,
                MuteConverter.INSTANCE
        );
        projectMuteService.run().join();
        var muteApi = new MuteApiStub(projectMuteService);

        var templateVarsFactory = new TemplateVarsFactory("solomon", "solomon.cloud");

        new TelegramUpdatesReceiver(
                telegramClient,
                timer,
                executor,
                config,
                dao,
                staffClient,
                distributedLock,
                resolver,
                telegramEventsDao,
                chartsClient,
                templateVarsFactory,
                muteApi);

        alert = ExpressionAlert.newBuilder()
                .setProjectId("solomon")
                .setId("alerting-cluster-membership")
                .setProgram("let foo = 42; alarm_if(foo > 10);")
                .setName("Alerting: Cluster membership")
                .build();

        multiAlert = ExpressionAlert.newBuilder()
                .setProjectId("solomon")
                .setId("fetch_errors")
                .setProgram("let foo = 42; alarm_if(foo > 10);")
                .setGroupByLabels(List.of("host", "type"))
                .setName("Fetcher: Fetch Errors")
                .build();

        subAlert1 = SubAlert.newBuilder()
                .setParent(multiAlert)
                .setGroupKey(Labels.of("host", "Sas", "type", "IPC_QUEUE_OVERFLOW"))
                .build();

        subAlert2 = SubAlert.newBuilder()
                .setParent(multiAlert)
                .setGroupKey(Labels.of("host", "Vla", "type", "IPC_QUEUE_OVERFLOW"))
                .build();

        var templateFactory = new MustacheTemplateFactory();
        var factory = new TelegramNotificationChannelFactory(
                telegramClient, staffClient, chartsClient, resolver,
                templateFactory, TelegramTemplate.create(templateFactory, templateVarsFactory),
                new RawTelegramLimits(10, 10, 0.6),
                timer, telegramEventsDao);

        resolver.addTelegramLogin(271780368L, "uran1x");
        resolver.addGroupTitle(-1001441039317L, "Telegram /dev/null");

        channel = factory.createChannel(TelegramNotification.newBuilder()
                .setId(UUID.randomUUID().toString())
                .setProjectId("solomon")
                //.setLogin("uranix")
                .setChatId(-1001441039317L)
                .setSendScreenshot(true)
                .build());

        actor = new ActorWithFutureRunner(this::sendEvent, ForkJoinPool.commonPool());

        /*
        actor = null;

        var keyboard = new InlineKeyboard(1);
        var chatId = 271780368L;
        keyboard.getRow(0).add(ButtonType.MUTE.with("---uuid---"));
        var r1 = telegramClient.sendMessage(chatId, "orig", ParseMode.HTML, keyboard).join();
        System.out.println(r1);
        var r2 = telegramClient.forceReply(chatId, "<a href=\"foo\">bar</a>", ParseMode.HTML, r1.getMessageId()).join();
        System.out.println(r2);
        Thread.sleep(5000);
        var r3 = telegramClient.replyKeyboardRemove(chatId).join();
        System.out.println(r3);
        */
    }

    private static Event makeEvent(Alert alert) {
        var codes = EvaluationStatus.Code.values();
        var status = codes[ThreadLocalRandom.current().nextInt(codes.length)];

        return new Event(alert, new AlertProcessingState(
                EvaluationState.newBuilder(alert)
                        .setSince(Instant.now())
                        .setLatestEval(Instant.now())
                        .setStatus(status.toStatus(Map.of("descr", """
                                        Sed ut perspiciatis, unde omnis iste natus error sit voluptatem accusantium doloremque laudantium,
                                        """), Map.of()))
                        .build(),
                new AlertMuteStatus(AlertMuteStatus.MuteStatusCode.NOT_MUTED, List.of())));
    }

    private CompletableFuture<Void> sendEvent() {
        logger.info("send event");
        if (ThreadLocalRandom.current().nextBoolean()) {
            return channel.send(Instant.now(), makeEvent(alert)).thenAccept(x -> {});
        } else {
            var event1 = makeEvent(subAlert1);
            var event2 = makeEvent(subAlert2);
            return CompletableFutures.allOf2(
                    channel.send(Instant.now(), event1).handle((r, t) -> null),
                    channel.send(Instant.now(), event2).handle((r, t) -> null)
            ).thenAccept(r -> {});
        }
    }

    private void run() {
        timer.scheduleWithFixedDelay(actor::schedule, 0, 10, TimeUnit.SECONDS);
    }

    public static void main(String[] args) throws IOException, InterruptedException {
        (new TelegramBot(args)).run();
    }
}
