#include <library/cpp/monlib/encode/format.h>
#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/monlib/metrics/metric_value.h>
#include <library/cpp/http/client/client.h>

#include <util/generic/hash.h>
#include <util/generic/singleton.h>
#include <util/stream/str.h>
#include <util/string/builder.h>
#include <util/system/hostname.h>
#include <util/system/spinlock.h>

#include <thread>
#include <cstring>

extern "C" {
    // inlined definitions & declarations from
    // #include "collectd.h"
    // #include "plugin.h"

    typedef uint64_t cdtime_t;
    #define DATA_MAX_NAME_LEN 128
    struct data_source_s {
      char name[DATA_MAX_NAME_LEN];
      int type;
      double min;
      double max;
    };
    typedef struct data_source_s data_source_t;

    struct data_set_s {
      char type[DATA_MAX_NAME_LEN];
      size_t ds_num;
      data_source_t *ds;
    };
    typedef struct data_set_s data_set_t;

    typedef unsigned long long counter_t;
    typedef double gauge_t;
    typedef int64_t derive_t;
    typedef uint64_t absolute_t;

    union value_u {
      counter_t counter;
      gauge_t gauge;
      derive_t derive;
      absolute_t absolute;
    };
    typedef union value_u value_t;

    struct value_list_s {
      value_t *values;
      size_t values_len;
      cdtime_t time;
      cdtime_t interval;
      char host[DATA_MAX_NAME_LEN];
      char plugin[DATA_MAX_NAME_LEN];
      char plugin_instance[DATA_MAX_NAME_LEN];
      char type[DATA_MAX_NAME_LEN];
      char type_instance[DATA_MAX_NAME_LEN];
      void *meta;
    };

    typedef struct value_list_s value_list_t;

    #define DS_TYPE_COUNTER 0
    #define DS_TYPE_GAUGE 1
    #define DS_TYPE_DERIVE 2
    #define DS_TYPE_ABSOLUTE 3

    struct user_data_s {
      void *data;
      void (*free_func)(void *);
    };
    typedef struct user_data_s user_data_t;

    struct oconfig_value_s {
      union {
        char *string;
        double number;
        int boolean;
      } value;
      int type;
    };
    typedef struct oconfig_value_s oconfig_value_t;

    struct oconfig_item_s;
    typedef struct oconfig_item_s oconfig_item_t;
    struct oconfig_item_s {
      char *key;
      oconfig_value_t *values;
      int values_num;

      oconfig_item_t *parent;
      oconfig_item_t *children;
      int children_num;
    };

    typedef int (*plugin_init_cb)(void);
    typedef int (*plugin_write_cb)(const data_set_t *, const value_list_t *,
                                   user_data_t *);
    typedef int (*plugin_shutdown_cb)(void);
    int plugin_register_init(const char *name, plugin_init_cb callback);
    int plugin_register_write(const char *name, plugin_write_cb callback,
        user_data_t const *user_data);
    int plugin_register_complex_config(const char *type,
        int (*callback)(oconfig_item_t *));
    int plugin_register_shutdown(const char *name, plugin_shutdown_cb callback);
}

namespace {
    using namespace NMonitoring;

    const TStringBuf HOSTNAME = GetHostName();
    constexpr TStringBuf PUSH_URL = "http://solomon-prestable.yandex-team.ru/push";

    struct TMetric: TMoveOnly {
        EMetricType Kind = EMetricType::UNKNOWN;
        TMetricTimeSeries Series;
    };

    class TSamples {
    public:
        void Write(TLabels&& labels, TMetric&& s) {
            auto g = Guard(Lock_);

            auto it = Data_.find(labels);
            if (it == Data_.end()) {
                std::tie(it, std::ignore) = Data_.emplace(std::move(labels), TMetric{
                    .Kind = s.Kind,
                    .Series = std::move(s.Series),
                });

                return;
            }

            Y_VERIFY_DEBUG(it->second.Kind == s.Kind);
            if (it->second.Kind != s.Kind) {
                return;
            }

            it->second.Series.CopyFrom(s.Series);
        }

        THashMap<TLabels, TMetric> Flush() {
            auto g = Guard(Lock_);
            auto h = std::move(Data_);
            return h;
        }

    private:
        TAdaptiveLock Lock_;
        THashMap<TLabels, TMetric> Data_;
    };

    struct IMetricProvider {
        virtual ~IMetricProvider() = default;
        virtual THashMap<TLabels, TMetric> GetSamples() noexcept = 0;
    };

    class TConsumer {
        static constexpr auto DEFAULT_PUSH_INTERVAL = TDuration::Seconds(15);

    public:
        TConsumer(IMetricProvider* provider,
            TString project,
            TString cluster,
            TString service,
            TDuration interval = DEFAULT_PUSH_INTERVAL
            )
            : Provider_{provider}
            , Interval_{interval}
            , Url_{TStringBuilder() << PUSH_URL << "?project=" << project << "&cluster=" << cluster << "&service=" << service}
        {
        }

        void Start() noexcept {
            Thread_.Reset(new std::thread{&TConsumer::Run, this});
        }

        void Run() noexcept {
            while (!Stopped_) {
                const auto now = TInstant::Now();
                if (now - LastFlush_ < Interval_) {
                    Sleep(TDuration::MilliSeconds(500));
                    continue;
                }

                auto data = Provider_->GetSamples();
                Send(std::move(data));
                LastFlush_ = now;
            }
        }

        void Stop() noexcept {
            Stopped_ = true;
            Thread_->join();
        }

    private:
        void WriteSeries(IMetricEncoder& encoder, EMetricType kind, TMetricTimeSeries& series) {
            switch (kind) {
                case EMetricType::GAUGE:
                    Y_VERIFY_DEBUG(series.GetValueType() == EMetricValueType::DOUBLE);
                    series.ForEach([&] (auto time, auto, auto point) {
                        encoder.OnDouble(time, point.AsDouble());
                    });
                    break;

                case EMetricType::RATE:
                    Y_VERIFY_DEBUG(series.GetValueType() == EMetricValueType::UINT64, "%d", series.GetValueType());
                    series.ForEach([&] (auto time, auto, auto point) {
                        encoder.OnUint64(time, point.AsUint64());
                    });
                    break;

                case EMetricType::COUNTER:
                    Y_VERIFY_DEBUG(series.GetValueType() == EMetricValueType::UINT64);
                    series.ForEach([&] (auto time, auto, auto point) {
                        encoder.OnUint64(time, point.AsUint64());
                    });
                    break;

                default:
                    Y_VERIFY_DEBUG(false);
            }
        }

        void Send(THashMap<TLabels, TMetric>&& data) {
            TStringStream ss;
            auto encoder = BufferedEncoderJson(&ss);

            encoder->OnStreamBegin();

            for (auto&& [labels, samples]: data) {
                encoder->OnMetricBegin(samples.Kind);

                encoder->OnLabelsBegin();
                for (auto&& label: labels) {
                    encoder->OnLabel(label.Name(), label.Value());
                }

                encoder->OnLabelsEnd();

                WriteSeries(*encoder, samples.Kind, samples.Series);

                encoder->OnMetricEnd();
            }

            encoder->OnStreamEnd();
            encoder->Close();

            NHttp::TFetchQuery q{Url_,
                NHttp::TFetchOptions{}.SetTimeout(Interval_ * 2)
                    .SetRetryCount(5)
                    .SetRetryDelay(TDuration::Seconds(1))
                    .SetContentType(TString{NFormatContenType::JSON})
                    .SetPostData(ss.Str())
            };

            NHttp::FetchAsync(q, [] (NHttpFetcher::TResultRef r) {
                Cerr << "push done with code: " << r->Code << " " << r->Data << Endl;
            });
        }

    private:
        std::atomic_bool Stopped_{false};
        TInstant LastFlush_;
        THolder<std::thread> Thread_;
        IMetricProvider* Provider_{nullptr};
        TDuration Interval_;
        TString Url_;
    };

    class TSolomonWriter final: public IMetricProvider {
    public:
        TSolomonWriter() {
        }

        ~TSolomonWriter() = default;

        THashMap<TLabels, TMetric> GetSamples() noexcept override {
            return Samples_.Flush();
        }

        int Write(const data_set_t *ds, const value_list_t *vl, user_data_t*) noexcept {
            Y_VERIFY_DEBUG(ds != nullptr && vl != nullptr);
            if (ds == nullptr || vl == nullptr) {
                return 0;
            }

            if (std::strncmp(ds->type, vl->type, std::strlen(ds->type)) != 0) {
                return -1;
            }

            TMetric metric;
            for (auto i = 0u; i < ds->ds_num; ++i) {
                auto&& source = ds->ds[i];

                metric.Kind = ConvertKind(source.type);
                metric.Series = ConvertSeries(*vl, source.type);
                auto labels = ConvertLabels(*vl);

                Samples_.Write(std::move(labels), std::move(metric));
            }

            return 0;
        }

        int Shutdown() noexcept {
            if (Consumer_) {
                Consumer_->Stop();
            }

            return 0;
        }

        int Init(TString project, TString cluster, TString service) {
            if (project.Empty() || cluster.Empty() || service.Empty()) {
                return 1;
            }

            Consumer_.Reset(new TConsumer{this, std::move(project), std::move(cluster), std::move(service)});
            Consumer_->Start();
            return 0;
        }

    private:
        void AddPoint(TMetricTimeSeries& series, const value_t& val, int type) {
            switch (type) {
                case DS_TYPE_GAUGE:
                    series.Add(TInstant::Zero(), val.gauge);
                    break;
                case DS_TYPE_COUNTER:
                    series.Add(TInstant::Zero(), ui64(val.counter));
                    break;
                case DS_TYPE_ABSOLUTE:
                    series.Add(TInstant::Zero(), val.absolute);
                    break;
                case DS_TYPE_DERIVE:
                    series.Add(TInstant::Zero(), ui64(val.derive));
                    break;
                default:
                    Y_FAIL("UNKNOWN type: %d", type);
            }
        }

        EMetricType ConvertKind(int type) {
            switch (type) {
                case DS_TYPE_GAUGE:
                    return EMetricType::GAUGE;
                case DS_TYPE_COUNTER:
                    return EMetricType::COUNTER;
                case DS_TYPE_ABSOLUTE:
                    return EMetricType::IGAUGE;
                case DS_TYPE_DERIVE:
                    return EMetricType::RATE;
                default:
                    Y_FAIL("UNKNOWN type: %d", type);
            }

            Y_UNREACHABLE();
        }

        TLabels ConvertLabels(const value_list_t& val) {
            TLabels labels;

            if (val.host[0] != '\0') {
                TStringBuf hostValue;
                if (std::strncmp(val.host, "localhost", 9) == 0) {
                    hostValue = HOSTNAME;
                } else {
                    hostValue = TStringBuf{val.host};
                }

                labels.Add(TStringBuf("host"), TStringBuf(hostValue));
            }

            labels.Add(TStringBuf("plugin"), TStringBuf(val.plugin));
            if (val.plugin_instance[0] != '\0') {
                labels.Add(TStringBuf("pluginInstance"), TStringBuf(val.plugin_instance));
            }

            labels.Add(TStringBuf("type"), TStringBuf(val.type));

            if (val.plugin_instance[0] != '\0') {
                labels.Add(TStringBuf("typeInstance"), TStringBuf(val.type_instance));
            }

            return labels;
        }

        TMetricTimeSeries ConvertSeries(const value_list_t& vl, int type) {
            TMetricTimeSeries series;

            for (auto i = 0u; i < vl.values_len; ++i) {
                /// XXX time?
                AddPoint(series, vl.values[i], type);
            }

            return series;
        }
    private:
        TSamples Samples_;
        THolder<TConsumer> Consumer_;
    };

    // plugin init
    int Init() {
        return 0;
    }

    int Write(const data_set_t *ds, const value_list_t *vl, user_data_t *ud) {
        return Singleton<TSolomonWriter>()->Write(ds, vl, ud);
    }

    int Shutdown() {
        Singleton<TSolomonWriter>()->Shutdown();
        return 0;
    }

    int SetConfigValue(oconfig_item_t* root) {
        TString project, cluster, service;

        for (auto j = 0; j < root->children_num; ++j) {
            oconfig_item_t& item = root->children[j];

            for (auto i = 0; i < item.values_num; ++i) {
                oconfig_value_t& val = item.values[i];
                if (item.values[i].type != 0) {
                    return -1;
                }

                const auto* key = item.key;
                const auto* value = val.value.string;

                if (TStringBuf{key} == TStringBuf("Project")) {
                    project = value;
                } else if (TStringBuf{key} == TStringBuf("Service")) {
                    service = value;
                } else if (TStringBuf{key} == TStringBuf("Cluster")) {
                    cluster = value;
                } else {
                    return -1;
                }
            }
        }

        return Singleton<TSolomonWriter>()->Init(project, cluster, service);
    }
} // namespace

extern "C" {
    void module_register(void) {
        plugin_register_init("solomon", Init);
        plugin_register_write("solomon", Write, nullptr);
        plugin_register_shutdown("solomon", Shutdown);
        plugin_register_complex_config("solomon", SetConfigValue);
    }
}
