#!/usr/bin/env python
# -*-coding: utf-8 -*-
# vim: sw=4 ts=4 expandtab ai

import re
import os
import sys
import time
import socket
import subprocess
import argparse
from select import select

from log import grcl_logger
from misc import lock, unlock, config_parser, log_init, write_monitoring


_PATH_TO_CONFIG = '/etc/yabs-graphite-client/graphite-client.cfg'


def main(config_file=_PATH_TO_CONFIG):
    config = config_parser(config_file)

    log_init(config['LOG_FILE'], config['LOG_LEVEL'])
    log = grcl_logger.manager.getLogger('GRCL.Main')

    # Set lock
    lock(config['LOCK_FILE'])

    # Stats for monitoring file
    mon_stat = {'exec_time': 0,
                'total_checks': 0,
                'sucess_checks': 0
                }

    # prepare checks task
    checks_queue = []
    results_list = []
    for check_type, check_path_list in [
            ('hamster',config['CHECK_DIR_LIST']),
            ('graphite', config['CLEAN_CHECK_DIR_LIST'])]:
        for chk_dir in check_path_list:
            if not os.path.exists(chk_dir):
                log.debug('Can\'t find path %s' % chk_dir)
                continue
            # Find all executable files in path
            for path, _dir_list, fl_list in os.walk(chk_dir):
                for fl_name in fl_list:
                    full_path = os.path.join(path, fl_name)
                    if os.path.isfile(full_path) and \
                            os.access(full_path, os.X_OK):
                        checks_queue.append((check_type, full_path))

    mon_stat['total_checks'] = len(checks_queue)
    log.debug('Find %d checks' % mon_stat['total_checks'])

    #if 0 < len(checks_queue) < config['PARALLEL_CHECKS_POOL_SIZE']:
    #    count = len(checks_queue)
    #else:
    #    count = config['PARALLEL_CHECKS_POOL_SIZE']

    full_proc_list = []
    start_time = time.time()
    for check_type, checks_path in checks_queue:
        full_proc_list.append({
            'proc':subprocess.Popen([checks_path], stdout=subprocess.PIPE,
                                    stderr=subprocess.PIPE, shell=True),
            'path':checks_path,
            'stdout':'',
            'stderr':'',
            'check_type':check_type,
            })

    # ... wait for ...
    procs_list = full_proc_list[:]
    while procs_list:
        for proc_struct in procs_list:
            stdout, _rwl, _xl = select([proc_struct['proc'].stdout], [], [], 0)
            stderr, _rwl, _xl = select([proc_struct['proc'].stderr], [], [], 0)
            if stdout:
                proc_struct['stdout'] += stdout[0].read()
            if stderr:
                proc_struct['stderr'] += stderr[0].read(256)
            # ... check process done sucessfull
            if proc_struct['proc'].poll() != None:
                procs_list.remove( proc_struct )
                log.debug('Run check %s done sucessfull' % proc_struct['path'])
            # ... timeout expired - kill check process
            elif (time.time() - start_time) >= config['MAX_CHECK_TIMEOUT']:
                # on terminate exit code != 0
                proc_struct['proc'].terminate()
                log.error('Kill check %s by timeout %d' %\
                        (proc_struct['path'], config['MAX_CHECK_TIMEOUT']))
                procs_list.remove( proc_struct )
            else:
                time.sleep(0.2)

    # if sucess exec check ...
    for proc_struct in full_proc_list:
        if proc_struct['proc'].returncode == 0:
            sucess_parse = False
            out = None
            try:
                check_out = proc_struct['stdout'].strip() + \
                        proc_struct['proc'].stdout.read().strip()
                if proc_struct['check_type'] == 'hamster':
                    out = check_out_parser(check_out, config)
                else:
                    out = check_out + '\n'
                sucess_parse = True
            except Exception, err:
                log.error('Can\'t parse out for check %s: %s' % \
                        (proc_struct['path'], str(err)) )
                log.error('Check out: %s' % check_out)
            # If we have some result ...
            if sucess_parse:
                if out:
                    # ... write check result to result list for main thread
                    results_list += out
                else:
                    log.warning('Strange... no out for check %s' % \
                            proc_struct['path'])
                mon_stat['sucess_checks'] += 1
            if sucess_parse and out:
                log.debug('Check %s sucessfull return data' % \
                        proc_struct['path'])
        else:
            log.error('Check %s exit code = %s, stderr: %s' %\
                      (proc_struct['path'], proc_struct['proc'].returncode,
                      # return only 512 bytes from stderr
                      str(proc_struct['stderr'].strip() + \
                          proc_struct['proc'].stderr.read(256).strip())[:512]
                            ) )
    check_end_time = time.time()

    for i in xrange(config['TRY_TO_SENDER']):
        send_status = results_to_sender(results_list, config)
        if send_status:
            log.debug('Sucess push data to sender from try %d' % i)
            break
        else:
            log.warning('Error send data. Try #%d' % i)
            time.sleep(5.0)

    if not send_status:
        log.critical("Can't send data to sender. Exiting.")
        sys.exit(1)

    # Calculate execution time.
    end_time = time.time()
    total_time = end_time - start_time
    check_time = check_end_time - start_time
    send_time = end_time - check_end_time
    log.debug('Check time:  %.3f; Send time: %.3f; Total time: %.3f sec' % \
            (check_time, send_time, total_time))

    # Write to monitoring file
    mon_stat['exec_time'] = total_time
    write_monitoring(config['MONITORING_FILE'], mon_stat, log)

    # Release lock
    unlock()


def results_to_sender(results_list, config):
    log = grcl_logger.manager.getLogger('GRCL.ToSender')
    status = False

    # Open sock from server
    sock = None
    try:
        if config['SENDER_LISTEN_PR'] == socket.AF_INET + socket.AF_INET6:
            sock_type = socket.AF_INET6
        else:
            sock_type = config['SENDER_LISTEN_PR']
        sock = socket.socket(sock_type, socket.SOCK_STREAM)
        sock.settimeout(1.0)
        sock.connect((config['SENDER_LISTEN_HOST'], config['SENDER_PORT']))
        log.debug('Send %d params to sender' % len(results_list))
        send_count = 0
        for metric in results_list:
            sock.send(metric)
            send_count += 1
        status = True
        log.debug('Sucess send %d params to sender' % send_count)
    except Exception, err:
        log.error('Error send data to sender: %s' % err.message)
    finally:
        if sock:
            sock.close()
    return status


def check_out_parser(out, cfg):
    """
    Function for convert hamster check output to graphite format
    """
    result = []
    # Workaround for some rare cases, when hamster check can be multiline
    out = out.replace("\n", '')
    for line in out.replace('{','').replace('}','').replace(']','').split('['):
        # pass blank line(data source)
        if not line:
            continue
        # Split for left and right side
        check_src, data = line.split('=')
        # Get all params from check out left side
        check_src = check_src.strip().replace('.','_')
        # Hostname and time is missing
        if re.search('^[a-zA-Z][^:]*?$', check_src):
            hostname = cfg['HOSTNAME']
            check_name = check_src
            # timestamp get on config init(start graphite-client) in misc module
            timestamp = cfg['TIMESTAMP']
        # Hostname is missing
        elif re.search('^[a-zA-Z][^:]*?:\d+$', check_src):
            # Resolve hostname on config init
            hostname = cfg['HOSTNAME']
            check_name, timestamp = check_src.split(':')
            timestamp = int(float(timestamp))
        # Time is missing
        elif re.search('^.+?:[a-zA-Z][^:]*?$', check_src):
            timestamp = cfg['TIMESTAMP']
            hostname, check_name = check_src.split(':')

        # Get all check values from checks out right side
        for field in data.split():
            field_name, value = field.split(':')
            field_name = field_name.strip().replace('.','_')
            value = value.strip()
            # If not scheme - use deault scheme and tree position
            if cfg['GRAPHITE_SCHEME']:
                send_line = '%s.%s.%s.%s %s %d\n' % \
                    (cfg['GRAPHITE_SCHEME'], hostname, check_name,
                    field_name, value, timestamp)
            else:
                send_line = '%s.%s.%s %s %d\n' % \
                            (hostname,
                             check_name,
                             field_name,
                             value,
                             timestamp)
            result.append(send_line)
    return result


def parse_cmdline_args():
    parser = argparse.ArgumentParser(description='Graphite client.')
    parser.add_argument('--config', default=_PATH_TO_CONFIG, help='path to config file')
    args, unknown = parser.parse_known_args()
    return args


if __name__ == '__main__':
    parsed_args = parse_cmdline_args()
    main(config_file=parsed_args.config)
    sys.exit(0)

