#!/usr/bin/env python
# -*-coding: utf-8 -*-
# vim: sw=4 ts=4 expandtab ai

import os
import sys
import socket
import fcntl
import time
import ConfigParser
import logging
from logging.handlers import WatchedFileHandler
import random
import json
import copy

from log import grcl_logger


_lock = None
_dns_cache = {}
DNS_CACHE_TIMEOUT = 3600
DNS_TM_DIFF = 300

# Default params for config
DEFAULTS = {
    'CHECK_DIR_LIST': '/usr/share/hamster-client/checks_available',
    'CLEAN_CHECK_DIR_LIST': '/usr/share/graphite-client/checks_available',
    'DROPPED_METRIC_FILE': '',
    'GLOBAL_METRIC_PREFIX': '',
    'GRAPHITE_SCHEME': 'one_min',
    'LOCK_FILE': '/var/lock/graphite-client.lock',
    'LOG_FILE': '/var/log/graphite-client/graphite-client.log',
    'LOG_LEVEL': '20',
    'MAX_CHECK_TIMEOUT': '40',
    'MAX_METRIC_COUNT': '500',
    'MAX_UNIQ_METRICS': '32000',
    'MONITORING_FILE': '/tmp/yabs-graphite-client-monitoring',
    'MONITORING_PREFIX': '""',
    'PARALLEL_CHECKS_POOL_SIZE': '10',
    'QUEUE_DIR': '/tmp/',
    'QUEUE_TIMEOUT': '864000000',
    'SENDER_LISTEN_PR': str(socket.AF_INET),
    'SENDER_LISTEN_HOST': 'localhost',
    'SENDER_LOG_FILE': '/var/log/graphite-client/graphite-sender.log',
    'SENDER_MONITORING_FILE': '/tmp/yabs-graphite-sender-monitoring',
    'SENDER_PID': '/run/yabs-graphite-sender.pid',
    'SENDER_PORT': '42000',
    'SENDER_TIMEOUT': '10.0',
    'TRY_TO_SENDER': '3',
    'UNIQ_MONITORING_FILE': '/tmp/yabs-graphite-sender-metrics',
    'SERVER_CHECK_PTR': 'yes',
}

def get_ip(host, log=None, check_ptr=True):
    """Function for get ip address from hostname"""

    # Get info from cache
    global _dns_cache
    # if host in local cache
    actual_cache_info = False
    if _dns_cache.has_key(host):
        ip_proto_list, fqdn, expire_tm = _dns_cache[host]
        if expire_tm > time.time():
            actual_cache_info = True

    fail = False

    if not actual_cache_info:
        # Try get ip from DNS
        try:
            addr = socket.getaddrinfo(host, 22, 0, 0, socket.SOL_TCP)
            ip_proto_list = [ (adr_dt[4][0], adr_dt[0]) for adr_dt in addr ]
        except socket.gaierror:
            fail = True

        # If get name need check it. Do reverse lookup
        # It code for check any.yandex.ru. Usually
        # not_exist_in_yandex_ru.yandex.ru resolv to any.yandex.ru ip. And
        # reverse resolving check it.
        if not fail and check_ptr:
            resolved_fqdn = ''
            for ip, proto in ip_proto_list:
                fqdn = socket.getfqdn(ip)
                if fqdn.startswith(host):
                    resolved_fqdn = fqdn
                    break
                else:
                    fail = True
                    if log:
                        log.warning('Error in reverse lookup: %s => %s => %s' % \
                                (host, ip, fqdn) )
                    else:
                        print 'Error in reverse lookup: %s => %s => %s' % \
                                (host, ip, fqdn)
            if resolved_fqdn:
                fail = False
                fqdn = resolved_fqdn
        elif not fail:
            fqdn = host

    if not fail:
        if not actual_cache_info:
            # Add info to cache
            _dns_cache[host] = (ip_proto_list, fqdn,
                    # cache values for ...
                    (time.time() + random.randint(-DNS_TM_DIFF, DNS_TM_DIFF) + DNS_CACHE_TIMEOUT))
    else:
        return [], ''

    return ip_proto_list, fqdn


def lock(lockfile):
    log = grcl_logger.manager.getLogger('GRCL.Lock')
    global _lock
    _lock = open(lockfile, "w")
    try:
        fcntl.lockf(_lock.fileno(), fcntl.LOCK_EX | fcntl.LOCK_NB)
        log.debug('Set lock to %s' % lockfile)
    except IOError:
        log.error('Can\'t set lock to %s. Exiting' % lockfile)
        sys.exit(1)


def unlock():
    """
    Function for remove file lock
    """
    log = grcl_logger.manager.getLogger('GRCL.Unlock')
    global _lock
    fcntl.lockf(_lock.fileno(), fcntl.LOCK_UN)
    log.debug('Release lock')
    _lock.close()


def config_parser(cfg_path):
    grcl_config = {}

    cfg = ConfigParser.ConfigParser(DEFAULTS)

    # Wait for config then installed
    for i in xrange(10):
        if not os.path.exists(cfg_path):
            time.sleep(3)
        else:
            break

    if not cfg.read(cfg_path):
        print 'Error reading config: %s' % cfg_path
        sys.exit(1)

    try:
        # Get timestamp:
        try:
            timestamp_delta = cfg.get('main',
                   'TIMESTAMP_DELTA').strip()
            grcl_config['TIMESTAMP'] = int(time.time()) + int(timestamp_delta)
        except:
            grcl_config['TIMESTAMP'] = int(time.time())
        #print "DEBUG: timestamp delta %s, timestamp %s" % (timestamp_delta, time.ctime(grcl_config['TIMESTAMP']))

        # Get options from config
        grcl_config['CHECK_DIR_LIST'] = cfg.get('main',
                'CHECK_DIR_LIST').strip().split(',')
        grcl_config['CLEAN_CHECK_DIR_LIST'] = cfg.get('main',
                'CLEAN_CHECK_DIR_LIST').strip().split(',')
        grcl_config['PARALLEL_CHECKS_POOL_SIZE'] = cfg.getint('main',
                'PARALLEL_CHECKS_POOL_SIZE')
        grcl_config['MAX_CHECK_TIMEOUT'] = cfg.getint('main',
                'MAX_CHECK_TIMEOUT')
        grcl_config['SERVER_LIST'] = cfg.get('main',
                'SERVER_LIST').strip().split(',')
        grcl_config['SERVER_CHECK_PTR'] = cfg.getboolean('main', 'SERVER_CHECK_PTR')
        grcl_config['GRAPHITE_SCHEME'] = cfg.get('main', 'GRAPHITE_SCHEME')
        grcl_config['LOCK_FILE'] = cfg.get('main', 'LOCK_FILE')
        grcl_config['LOG_FILE'] = cfg.get('main', 'LOG_FILE')
        grcl_config['LOG_LEVEL'] = cfg.getint('main', 'LOG_LEVEL')
        grcl_config['QUEUE_DIR'] = cfg.get('main', 'QUEUE_DIR')
        grcl_config['MONITORING_PREFIX'] = cfg.get('main', 'MONITORING_PREFIX')
        grcl_config['QUEUE_TIMEOUT'] = cfg.getint('main', 'QUEUE_TIMEOUT')
        grcl_config['MONITORING_FILE'] = cfg.get('main', 'MONITORING_FILE')
        grcl_config['GLOBAL_METRIC_PREFIX'] = cfg.get('main',
                'GLOBAL_METRIC_PREFIX')
        grcl_config['SENDER_MONITORING_FILE'] = cfg.get('main',
                'SENDER_MONITORING_FILE')
        grcl_config['SENDER_PORT'] = cfg.getint('main', 'SENDER_PORT')
        grcl_config['MAX_METRIC_COUNT'] = cfg.getint('main', 'MAX_METRIC_COUNT')
        grcl_config['SENDER_TIMEOUT'] = cfg.getfloat('main', 'SENDER_TIMEOUT')
        grcl_config['SENDER_LOG_FILE'] = cfg.get('main', 'SENDER_LOG_FILE')
        grcl_config['SENDER_PID'] = cfg.get('main', 'SENDER_PID')
        grcl_config['TRY_TO_SENDER'] = cfg.getint('main', 'TRY_TO_SENDER')
        grcl_config['SENDER_LISTEN_PR'] = cfg.getint('main',
                'SENDER_LISTEN_PR')
        grcl_config['SENDER_LISTEN_HOST'] = cfg.get('main',
                'SENDER_LISTEN_HOST')
        grcl_config['DROPPED_METRIC_FILE'] = cfg.get('main',
                'DROPPED_METRIC_FILE')
        grcl_config['MAX_UNIQ_METRICS'] = cfg.getint('main', 'MAX_UNIQ_METRICS')
        grcl_config['UNIQ_MONITORING_FILE'] = cfg.get('main', 'UNIQ_MONITORING_FILE')

        # Get hostname
        if cfg.has_option('main', 'HOSTNAME'):
            grcl_config['HOSTNAME'] = cfg.get('main', 'HOSTNAME')
        else:
            _ip_proto_list, fqdn = get_ip(socket.gethostname())
            if not fqdn:
                print 'Error - can\'t get hostname. Try to use HOSTNAME option in config.'
                sys.exit(1)
            else:
                grcl_config['HOSTNAME'] = fqdn.replace('.','_')

    except (ConfigParser.NoSectionError, ConfigParser.NoOptionError), err:
        print 'Error parse config %s: %s' % (cfg_path, err.message)
        sys.exit(1)

    return grcl_config


def log_init(log_file, log_level):
    """
    Function for init log file
    """
    # Log Handler. Set logfile
    ch = WatchedFileHandler(log_file)
    # create formatter for log
    formatter = logging.Formatter('%(asctime)s %(name)-20s %(levelname)-8s %(message)s')
    # add formatter to handler
    ch.setFormatter(formatter)
    grcl_logger.addHandler(ch)
    # Set log level from config
    grcl_logger.setLevel(log_level)


def write_uniq(fpath, data, log):
    if os.path.exists(fpath):
        old_data = None
        try:
            f = open(fpath)
            old_data = set([el.strip() for el in f.readlines()])
            f.close()
        except IOError, err:
            log.error("Cant't load data from file %s" % str(err))
        new_data = data
        if old_data:
            new_data = new_data.union(old_data)
        try:
            f = open(fpath, 'w')
            f.write('\n'.join(list(new_data)))
            f.close()
        except IOError, err:
            log.error("Cant't write to monitoring file %s" % str(err))
    else:
        try:
            f = open(fpath, 'w')
            f.write('\n'.join(list(data)))
            f.close()
        except IOError, err:
            log.error("Cant't write to monitoring file %s" % str(err))
    return


def write_monitoring(fpath, data, log):
    # Note: do not delete fpath while graphite is running (renaming to it is ok since rename is atomic on *nix):
    # graphite-client.py checks some files written by this function
    # If these files are missing, it does not perform graphite-client checks at all.
    try:
        tmppath = fpath + '.tmp'
        f = open(tmppath, 'w')
        f.write(json.dumps(data))
        f.close()
        os.rename(tmppath, fpath)
    except IOError, err:
        log.error("Cant't write to monitoring file %s" % str(err))
    return



