import os
import re
import time
import json

import system

try:
    import yt.wrapper as yt
    import yt.packages.requests as req
    NO_YT = False
except ImportError:
    NO_YT = True


#=============================================================================================================


YDB 		= "/usr/local/bin/ydb"
KV  		= "/usr/local/bin/kv"
YDB_ENDPOINT 	= "grpc://localhost:2135"
S3_KEYFILE 	= "/Berkanavt/keys/solomon/mds.secret"
S3_ENDPOINT 	= "s3.mds.yandex.net"


#=============================================================================================================

def YdbCmd(Endpoint, DB, SaKeyFile):
    Cmd = [YDB, "-e", Endpoint, "-d", DB]
    if SaKeyFile:
        Cmd += ["--sa-key-file", SaKeyFile]
    if DB.startswith("/pre-prod_global"):
        Cmd += ["--iam-endpoint", "iam.api.cloud-preprod.yandex.net"]
    return Cmd

def YdbTableParser(Data):
    Head       = None
    NonTable   = []
    Table      = []
    Lines      = [L.strip() for L in Data.split('\n')]
    for Line in Lines:
        if len(Line) == 0 or Line[0] == "\xe2":
            continue
        elif Line[0] != '|':
            NonTable.append(Line)
        elif not Head:
            Head = [F.strip() for F in Line.split('|') if len(F)]
        else:
            Row  = [F.strip() for F in Line.split('|') if len(F)]
            if len(Row) == len(Head):
                Table.append(dict(zip(Head, Row)))
            elif len(Row) == 1:
                if len(Table) > 0:
                    if None in Table[-1]:
                        Table[-1][None].append(Row[0])
                    else:
                        Table[-1][None] = [Row[0]]
            else:
                system.Log("Bad row in ydb table (header=%s): %s" % (Head, Line))
    return Table, NonTable


def YdbList(DB, Path, Recursive=False, Endpoint=YDB_ENDPOINT, SaKeyFile=None):
    Result = {"dir":[], "table":[]}
    E = system.Exec(YdbCmd(Endpoint, DB, SaKeyFile) + ["scheme", "ls", "-lR" if Recursive else "-l", Path])
    for Row in YdbTableParser(E.out)[0]:
        if Row.get("Type") in Result:
            Result[Row.get("Type")].append(Row.get("Name"))
    return Result


def YdbTree(DB, Path, SkipList=[], Endpoint=YDB_ENDPOINT, SaKeyFile=None):
    SkipList = ["/%s" % S.strip('/') for S in SkipList]
    Dict = YdbList(DB, Path, True, Endpoint, SaKeyFile)
    for Table in Dict["table"]:
        TablePath = os.path.join(DB, Path, Table)
        Skip = False
        for E in SkipList:
            if E in TablePath:
                Skip = True
                break
        if not Skip:
            yield TablePath


def YdbDropTable(DB, Path, Endpoint=YDB_ENDPOINT, SaKeyFile=None):
    E = system.Exec(YdbCmd(Endpoint, DB, SaKeyFile) + ["table", "drop", Path])
    if len(E.err) > 0:
        system.Log("Error while drop %s: %s" % (Path, repr(E.err)))
    return E.status


def YdbRmdir(DB, Path, Endpoint=YDB_ENDPOINT, SaKeyFile=None):
    E = system.Exec(YdbCmd(Endpoint, DB, SaKeyFile) + ["scheme", "rmdir", Path])
    if len(E.err) > 0:
        system.Log("Error while rmdir %s: %s" % (Path, repr(E.err)))
    return E.status


ReOldBackup = re.compile("^~backup_([0-9]{8}T[0-9]{6})$")
def YdbRmOldBackup(DB, Path, MinAge=14400, Endpoint=YDB_ENDPOINT, SaKeyFile=None):
    Time = time.time()
    for Name in YdbList(DB, Path, False, Endpoint, SaKeyFile)["dir"]:
        M = ReOldBackup.match(Name)
        if not M:
            system.Log("Item '%s' not removed: does not match backup dir regex" % Name)
            continue
        Fail = False
        if Time - time.mktime(time.strptime(M.group(1), "%Y%m%dT%H%M%S")) > MinAge:
            FullName = os.path.join(Path, Name)
            for Table in YdbList(DB, FullName, False, Endpoint, SaKeyFile)["table"]:
                FullTableName = os.path.join(FullName, Table)
                system.Log("Removing old backup table %s" % FullTableName)
                if YdbDropTable(DB, FullTableName, Endpoint, SaKeyFile) != 0:
                    Fail = True
                    break
            if not Fail:
                system.Log("Removing old backup dir %s" % FullName)
                YdbRmdir(DB, FullName, Endpoint, SaKeyFile)
        else:
            system.Log("%s not removed: too soon to remove" % Name)


def SplitFile(File, FsPath, MaxSize=2147483648):
    ChunkSize   = 8192
    FileCount   = 0
    FsPathMod   = FsPath
    ChunkLength = ChunkSize
    while ChunkLength > 0:
        FileFd     = os.open(FsPathMod, os.O_CREAT | os.O_TRUNC | os.O_WRONLY, 0644)
        FileCount += 1
        FileSize   = 0
        while FileSize < MaxSize:
            Chunk = File.read(ChunkSize)
            ChunkLength = len(Chunk)
            if ChunkLength == 0:
                break
            os.write(FileFd, Chunk)
            FileSize += ChunkLength
        os.close(FileFd)
        FsPathMod = "%s.%03d" % (FsPath, FileCount)
    if FileCount > 1:
        system.Exec(["/bin/mv", FsPath, "%s.000" % FsPath])
        return ["%s.%03d" % (FsPath, C) for C in xrange(FileCount)]
    return [FsPath]


def YdbTableToTgzSplitFile(DB, YdbFilePath, FsFileDir, MaxSize=2147483648, Endpoint=YDB_ENDPOINT, SaKeyFile=None):
    TableName   = os.path.basename(YdbFilePath)
    FsDumpDir   = os.path.join(FsFileDir, TableName)
    FileList    = []
    E = system.Exec(YdbCmd(Endpoint, DB, SaKeyFile) + ["tools", "dump", "-p", YdbFilePath, "-o", FsDumpDir])
    if E.status == 0 and not os.path.exists(os.path.join(FsDumpDir, TableName, "incomplete")):
        system.Log("Successfully dumped ydb:%s -> %s" % (YdbFilePath, FsDumpDir))
        pTgz     = system.ExecPipe(["/bin/tar", "czf", "-", "-C", FsDumpDir, "."])
        FileList = SplitFile(pTgz.stdout, "%s.tgz" % FsDumpDir, MaxSize)
        DumpSize = system.Du(FsDumpDir)
    else:
        system.Log("Failed to dump ydb:%s -> %s" % (YdbFilePath, FsDumpDir))
        DumpSize = 0
    system.Exec(["/bin/rm", "-r", FsDumpDir])
    return FileList, DumpSize


def YdbExportToYt(DB, SrcToDstDict, YtProxy, Endpoint=YDB_ENDPOINT, SaKeyFile=None):
    Cmd = YdbCmd(Endpoint, DB, SaKeyFile) + ["export", "yt", "--proxy", YtProxy, "--use-type-v3", "--retries", "20", "--format", "proto-json-base64"]
    for SrcPath, DstPath in SrcToDstDict.items():
        Cmd += ["--item", "source=%s,destination=%s" % (SrcPath, DstPath)]
    E = system.Exec(Cmd, env={"LANG": "C", "YT_TOKEN": os.environ["YT_TOKEN"]})
    if E.status != 0:
        system.Log("Error while starting export of %s: %s" % (SrcToDstDict, repr(E.err)))
    else:
        try:
            Result = json.loads(E.out)
            if str(Result.get("status")) == "SUCCESS":
                return str(Result["id"]) if "id" in Result else None
        except Exception as e:
            system.Log("Got exception while starting ydb export to yt (%s -> %s): %s" % (SrcToDstDict.keys(), SrcToDstDict.values(), e))


def YdbExportsList(DB, Endpoint=YDB_ENDPOINT, SaKeyFile=None):
    E = system.Exec(YdbCmd(Endpoint, DB, SaKeyFile) + ["operation", "list", "export", "--format", "proto-json-base64", "--page-size", "1000"])
    if E.status != 0:
        system.Log("Error while listing exports: %s" % (repr(E.err)))
    else:
        try:
            Result = json.loads(E.out)
            return [{
                "status":    str(P["status"]).strip(),               # SUCCESS
                "ready":     P.get("ready", False),
                "id":        str(P["id"]).strip(),
                "progress":  str(P["metadata"]["progress"]).strip(), # PROGRESS_DONE
                "items":     {str(I["sourcePath"]).strip(): str(I["destinationPath"]).strip() for I in P["metadata"]["settings"]["items"]}
            } for P in Result.get("operations", [])]
        except Exception as e:
            system.Log("Got exception while listing ydb exports to yt: %s" % e)
    return []


def YdbOperationGet(DB, Id, Endpoint=YDB_ENDPOINT, SaKeyFile=None):
    E = system.Exec(YdbCmd(Endpoint, DB, SaKeyFile) + ["operation", "get", "--format", "proto-json-base64", Id])
    if E.status != 0:
        system.Log("Error while getting ydb operation (id=%s): %s" % (Id, repr(E.err)))
    else:
        try:
            P = json.loads(E.out)
            return {
                "status":    str(P["status"]),               # SUCCESS
                "ready":     P.get("ready", False),
                "id":        str(P["id"]),
                "progress":  str(P["metadata"]["progress"]), # PROGRESS_DONE
                "items":     {str(I["sourcePath"]): str(I["destinationPath"]) for I in P["metadata"]["settings"]["items"]}
            }
        except Exception as e:
            system.Log("Got exception while getting ydb operation (id=%s): %s" % (Id, e))
    return {}


def YdbOperationForget(DB, Id, Endpoint=YDB_ENDPOINT, SaKeyFile=None):
    E = system.Exec(YdbCmd(Endpoint, DB, SaKeyFile) + ["operation", "forget", Id])
    if E.status != 0:
        system.Log("Error while forgetting operation (id=%s): %s" % (Id, repr(E.err)))
        return False
    return True


def YdbOperationCancel(DB, Id, Endpoint=YDB_ENDPOINT, SaKeyFile=None):
    E = system.Exec(YdbCmd(Endpoint, DB, SaKeyFile) + ["operation", "cancel", Id])
    if E.status != 0:
        system.Log("Error while cancelling operation (id=%s): %s" % (Id, repr(E.err)))
        return False
    return True


def YdbExportsCleanup(DB, Endpoint=YDB_ENDPOINT, SaKeyFile=None):
    for P in YdbExportsList(DB, Endpoint, SaKeyFile):
        if P["progress"] in ["PROGRESS_DONE", "PROGRESS_CANCELLED"]:
            system.Log("Clenaning up export id=%s status=% ready=%s progress=%s" % (P["id"], P["status"], P["ready"], P["progress"]))
            if not YdbOperationForget(DB, P["id"], Endpoint, SaKeyFile):
                return False
    return True


def YdbOperationWait(DB, Id, Sleep=5, MaxFails=10, Verbose=True, Endpoint=YDB_ENDPOINT, SaKeyFile=None):
    while MaxFails > 0:
        P = YdbOperationGet(DB, Id, Endpoint, SaKeyFile)
        if len(P) == 0:
            MaxFails -= 1
        else:
            if P["ready"] is True and P["status"] == "SUCCESS" and P["progress"] == "PROGRESS_DONE":
                break
            if Verbose:
                system.Log("Transfer (id=%s): ready=%s status=%s progress=%s. Sleeping %d seconds..." % (P["id"], P["ready"], P["status"], P["progress"], Sleep))
        time.sleep(Sleep)
    else:
        if Verbose:
            system.Log("Waiting for tansfer failed (id=%s): ready=%s status=%s progress=%s" % (P.get("id"), P.get("ready"), P.get("status"), P.get("progress")))
        return False
    if Verbose:
        system.Log("Transfer is done (id=%s): ready=%s status=%s progress=%s" % (P["id"], P["ready"], P["status"], P["progress"]))
    YdbOperationCancel(DB, Id, Endpoint, SaKeyFile)
    return YdbOperationForget(DB, Id, Endpoint, SaKeyFile)


#=============================================================================================================


ReKVLs = re.compile("^([0-9]+) +([0-9]+) +([a-z0-9.-]+) +([0-9T:-]+)(\.[0-9]+Z)$")
def KVLs(Path):
    Cmd = [KV, "ls", "--path", Path]
    E = system.Exec(Cmd)
    Result = []
    for F in E.out.split("\n"):
        M = ReKVLs.match(F)
        if not M:
            continue
        Result.append({
            "Shard": M.group(1),
            "Tablet": M.group(2),
            "Host": M.group(3),
            "Time": time.mktime(time.strptime(M.group(4), "%Y-%m-%dT%H:%M:%S"))
        })
    return Result


def KVdump(YdbPath, ShardID, S3Prefix, S3KeyFile=S3_KEYFILE, S3Endpoint=S3_ENDPOINT, BalancerBypass=True, AmpFactor=1):
    Cmd = [
        KV,
        "s3dump",
        "--key-file", S3KeyFile,
        "--s3-enpoint", S3Endpoint,
        "--path", YdbPath,
        "--prefix", S3Prefix,
        "--shard", ShardID,
        "-q",
        "--fast",
        "--safety-factor", "0.1",
        "--amp", str(AmpFactor),
        "--stats"
    ]
    if BalancerBypass:
        Cmd.insert(6, "--balancer-bypass")
    E = system.ExecYield(Cmd)
    for L in E.read():
        if L.startswith("STATS"):
            yield L
        else:
            system.Log("%s" % L)
    if E.status == 0:
        yield "STATUS OK"
    elif E.status == 1:
        system.Log("Error in KVDump (%s): %s" % (E.status, E.p.stderr.read().strip()))
        yield "STATUS BAD"
    else:
        system.Log("Error in KVDump (%s): %s" % (E.status, E.p.stderr.read().strip()))
        yield "STATUS RETRY"


#=============================================================================================================


ReS3Ls = re.compile("^(.+[^ ]) +([0-9]+) +([0-9TZ:-]+)$")
def S3Ls(Prefix, S3KeyFile=S3_KEYFILE, S3Endpoint=S3_ENDPOINT, Recursive=False):
    Cmd = [KV, "s3ls", "--key-file", S3KeyFile, "--s3-enpoint", S3Endpoint, "-q", "--prefix", Prefix]
    if Recursive:
        Cmd += ["--recursive"]
    E = system.Exec(Cmd)
    Result = []
    for F in E.out.split("\n"):
        M = ReS3Ls.match(F)
        if not M:
            continue
        Result.append({
            "Name": M.group(1),
            "Size": int(M.group(2)),
            "Time": time.mktime(time.strptime(M.group(3), "%Y-%m-%dT%H:%M:%SZ"))
        })
    return Result


ReS3DirLs = re.compile("^(.+[^ ]) +<dir>$")
def S3DirLs(Prefix, S3KeyFile=S3_KEYFILE, S3Endpoint=S3_ENDPOINT):
    Cmd = [KV, "s3ls", "--key-file", S3KeyFile, "--s3-enpoint", S3Endpoint, "-q", "--prefix", Prefix]
    E = system.Exec(Cmd)
    Result = []
    for F in E.out.split("\n"):
        M = ReS3DirLs.match(F)
        if not M:
            continue
        Result.append(M.group(1))
    return Result


def S3Rm(List, S3KeyFile=S3_KEYFILE, S3Endpoint=S3_ENDPOINT):
    Result = True
    Bunch = 100
    Cmd = [KV, "s3rm", "--key-file", S3KeyFile, "--s3-enpoint", S3Endpoint, "-q"]
    for IndexStart in range(0, len(List), Bunch):
        E = system.Exec(Cmd + [List[F] for F in range(IndexStart, min(IndexStart + Bunch, len(List)))])
        if len(E.err) != 0:
            system.Log(E.err)
        if E.status != 0:
            Result = False
    return Result


def S3Put(FilePath, DestPath, S3KeyFile=S3_KEYFILE, S3Endpoint=S3_ENDPOINT, BalancerBypass=True):
    if not os.path.exists(FilePath):
        system.Log("No such file '%s' to write to s3" % FilePath)
        return False
    Cmd = [KV, "s3write", "--key-file", S3KeyFile, "--s3-enpoint", S3Endpoint, "-q", "--input", FilePath, DestPath]
    if BalancerBypass:
        Cmd.insert(6, "--balancer-bypass")
    E = system.Exec(Cmd)
    if len(E.err) != 0:
        system.Log(E.err)
    return (E.status == 0)


def S3Read(FilePath, S3KeyFile=S3_KEYFILE, S3Endpoint=S3_ENDPOINT):
    Cmd = [KV, "s3read", "--key-file", S3KeyFile, "--s3-enpoint", S3Endpoint, "-q", FilePath]
    E = system.ExecPipe(Cmd)
    return E.stdout.read()


def S3Write(Data, FilePath, S3KeyFile=S3_KEYFILE, S3Endpoint=S3_ENDPOINT):
    Cmd = [KV, "s3write", "--key-file", S3KeyFile, "--s3-enpoint", S3Endpoint, "-q", FilePath]
    E = system.ExecPipe(Cmd, stdin=True)
    E.stdin.write(Data)
    E.stdin.flush()
    E.stdin.close()
    E.wait()
    return (E.status == 0)


def S3RmOldFiles(Prefix, Age, S3KeyFile=S3_KEYFILE, S3Endpoint=S3_ENDPOINT, Recursive=True):
    MinTime = time.time() - Age
    RmList = [F["Name"] for F in S3Ls(Prefix, S3KeyFile=S3KeyFile, S3Endpoint=S3Endpoint, Recursive=Recursive) if F["Time"] < MinTime]
    if len(RmList) > 0:
        system.Log("Removing old S3 files: %s" % RmList)
        return S3Rm(RmList, S3KeyFile=S3KeyFile, S3Endpoint=S3Endpoint)
    return True


def S3RmByPrefix(Prefix, S3KeyFile=S3_KEYFILE, S3Endpoint=S3_ENDPOINT, Recursive=True):
    RmList = [F["Name"] for F in S3Ls(Prefix, S3KeyFile=S3KeyFile, S3Endpoint=S3Endpoint, Recursive=Recursive)]
    if len(RmList) > 0:
        system.Log("Removing S3 files: %s" % RmList)
        return S3Rm(RmList, S3KeyFile=S3KeyFile, S3Endpoint=S3Endpoint)
    return True


#=============================================================================================================


class TYt(object):
    def __init__(self, Proxy, Token):
        yt.config.set_proxy(Proxy)
        yt.config["token"] = Token

    def Cmd(self, Quiet, Command, *Args, **KwArgs):
        try:
            C = Command(*Args, **KwArgs)
            return C
        except yt.errors.YtTokenError as e:
            system.Log("Yt token error")
            return False
        except yt.errors.YtError as e:
            if not Quiet:
                system.Log("Yt error (%s): %s" % (e.code, [(E.get("code"), E.get("message")) for E in e.inner_errors]))
            return False
        except req.exceptions.RequestException as e:
            system.Log("Yt error: %s" % e.message)
            return False

    def Create(self, Path):
        return self.Cmd(False, yt.create, "table", Path) is not False

    def Exist(self, Node):
        return str(self.Cmd(True, yt.exists, Node)) == "true"

    def List(self, Path):
        return json.loads(self.Cmd(True, yt.list, Path, attributes=["uncompressed_data_size", "creation_time", "type"], format="json"))

    def Get(self, Path):
        return json.loads(self.Cmd(True, yt.get, Path, attributes=["uncompressed_data_size", "creation_time", "type"], format="json"))

    def Remove(self, NodeList):
        for Node in [NodeList] if isinstance(NodeList, str) else NodeList:
            if self.Cmd(False, yt.remove, Node) is False:
                return False
        return True

    def Rename(self, From, To):
        if not self.Exist(From):
            return False
        if self.Exist(To) and not self.Remove(To):
            return False
        return self.Cmd(False, yt.move, From, To) is not False

    def Merge(self, FromList, To, Mode="unordered"):
        M = self.Cmd(False, yt.run_merge, FromList, To, mode=Mode)
        return M is not False and M.get_state() == "completed"

    def Sort(self, FromList, To, SortBy):
        M = self.Cmd(False, yt.run_sort, FromList, To, sort_by=SortBy)
        return M is not False and M.get_state() == "completed"

    def Transaction(self):
        return yt.transaction.Transaction()

    def Lock(self, Path):
        if not self.Exist(Path) and not self.Create(Path):
            return False
        return self.Cmd(False, yt.lock, Path) is not False

    def UnLock(self, Path):
        if not self.Exist(Path):
            return False
        return self.Cmd(False, yt.unlock, Path) is not False
