#!/usr/bin/env python

import os
import re
import sys
import math
import fcntl
import socket
import syslog
import urllib2
import subprocess
import time
import datetime
import random
import struct
import hashlib
import threading
import Queue


# ==============================================================================================================


def Timed(Func):
    def F(*args, **kwargs):
        T = time.time()
        r = Func(*args, **kwargs)
        print("Function %s was running for %.6f sec" % (Func.__name__, time.time() - T))
        return r
    return F


# ==============================================================================================================


def NormalRand(Number):
    """ Return value has the same type as input value
    """
    N = struct.unpack("Q", os.urandom(8))[0]
    return (Number*N)/(1<<64)


def Random(Number):
    """ Return value is float
    """
    return Number*random.random()


reInt = re.compile("(-?[0-9]+)")
def Int(Str, Default=0):
    S = reInt.search(Str)
    return int(S.group(1)) if S else Default


reFloat = re.compile("(-?[0-9\.]+(e[+-]?[0-9]+)?)")
def Float(Str, Default=0.0):
    S = reFloat.search(Str)
    return float(S.group(1)) if S else Default


reBinarySuffixes = re.compile(r"\b([0-9.]+)([EPTGMKBb])\b")
BinarySuffixesDict = {"E":2**60,"P":2**50,"T":2**40,"G":2**30,"M":2**20,"K":2**10,"B":1,"b":1}
def RemoveBinarySuffixes(String):
    while True:
        Match = reBinarySuffixes.search(String)
        if Match is None:
            return String
        Number, Suffix = Match.group(1, 2)
        Number = float(Number)*BinarySuffixesDict[Suffix]
        StrNumber = ("%d" if Suffix == Suffix.upper() else "%e") % Number
        String = String[:Match.start()] + StrNumber + String[Match.end():]


reDecimalSuffixes = re.compile(r"\b([0-9.]+)([EPTGMKBbmunp])\b")
DecimalSuffixesDict = {"E":1e18,"P":1e15,"T":1e12,"G":1e9,"M":1e6,"K":1e3,"B":1,"b":1,"m":1e-3,"u":1e-6,"n":1e-9,"p":1e-12}
def RemoveDecimalSuffixes(String):
    while True:
        Match = reDecimalSuffixes.search(String)
        if Match is None:
            return String
        Number, Suffix = Match.group(1, 2)
        Number = float(Number)*DecimalSuffixesDict[Suffix]
        StrNumber = ("%d" if Suffix == Suffix.upper() else "%e") % Number
        String = String[:Match.start()] + StrNumber + String[Match.end():]


reOrder = re.compile(r"\b-?([0-9.]+)\b")
OrderSuffixesDict = {18:"E", 15:"P", 12:"T", 9:"G", 6:"M", 3:"K", 0:"", -3:"m", -6:"u", -9:"n", -12:"p"}
def AddCompressSuffix(String, Precision=1, Binary=False):
    SearchStart = 0
    MultDict = BinarySuffixesDict if Binary else DecimalSuffixesDict
    while True:
        Match = reOrder.search(String, SearchStart)
        if Match is None:
            return String
        Number = float(Match.group(1))
        Order  = (int(math.log10(Number if Number > 0 else 1) - (1 if Number < 1 and Number > 0 else 0))/3)*3
        Suffix = OrderSuffixesDict[Order]
        Mult   = MultDict.get(Suffix, 1)
        StrNumber = "%s%s" % (round(Number/Mult, Precision), Suffix)
        if StrNumber.endswith(".0"):
            StrNumber = StrNumber[:-2]
        String = String[:Match.start()] + StrNumber + String[Match.end():]
        SearchStart = Match.start() + len(StrNumber)


def CleanContent(Item):
    def CleanDict(Dict):
        ResultDict = {}
        for Key in Dict:
            Value = Dict[Key]
            if isinstance(Value, dict):
                Value = CleanDict(Value)
            elif isinstance(Value, list):
                Value = CleanList(Value)
            try:
                if Value is None or len(Value) == 0:
                    continue
            except TypeError:
                pass
            ResultDict[Key] = Value
        return ResultDict
    def CleanList(List):
        ResultList = []
        for Item in List:
            if isinstance(Item, dict):
                Item = CleanDict(Item)
            elif isinstance(Item, list):
                Item = CleanList(Item)
            try:
                if Item is None or len(Item) == 0:
                    continue
            except TypeError:
                pass
            ResultList.append(Item)
        return ResultList
    if isinstance(Item, dict):
        return CleanDict(Item)
    elif isinstance(Item, list):
        return CleanList(Item)
    else:
        return Item


# ==============================================================================================================


class ThreadedLog(object):
    def __init__(self, Logger, Tag=False):
        self.Logger = Logger
        self.Tag    = Tag
        self.Queue  = Queue.Queue(maxsize=1000)
        self.Thread = threading.Thread(target=self.LogThread)
        self.Thread.daemon = True
        self.Thread.start()

    def __call__(self, s):
        if self.Tag:
            s = "[%s] %s" % (threading.current_thread().name, s)
        try:
            self.Queue.put(s)
        except RuntimeError:
            self.Logger("%s (RuntimeError in ThreadedLog. Logging from main thread)" % s)

    def LogThread(self):
        while True:
            self.Logger(self.Queue.get())
            self.Queue.task_done()


def OutLog(s):
    print("%s - %s" % (datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S.%f"), s))


class FileLog(object):
    def __init__(self, Filename, MaxSize=10*1024*1024, Count=10, Archive=False):
        self.Filename = Filename
        self.MaxSize  = MaxSize
        self.Count    = Count
        self.Archive  = Archive
        self.SpawnPid = None
        self.Size     = 0
        self.Fileno   = -1
        self.Sync     = False

    def CheckRotate(self):
        if self.Size >= self.MaxSize:
            if not self.Sync:
                os.fsync(self.Fileno)
            os.close(self.Fileno)
            self.Fileno = -1
            for C in range(self.Count, 1, -1):
                NewName = "%s.%d%s" % (self.Filename, C, ".gz" if self.Archive else "")
                OldName = "%s.%d%s" % (self.Filename, C - 1, ".gz" if self.Archive else "")
                if os.path.exists(OldName):
                    os.rename(OldName, NewName)
            FirstFile = "%s.1" % self.Filename
            os.rename(self.Filename, FirstFile)
            if self.Archive:
                self.SpawnPid = Spawn(["/bin/gzip", "-S", ".gz", FirstFile])
        if self.SpawnPid:
            try:
                if os.waitpid(self.SpawnPid, os.WNOHANG)[0] == self.SpawnPid:
                    self.SpawnPid = None
            except OSError:
                self.SpawnPid = None
        if self.Fileno < 0:
            self.Fileno = os.open(self.Filename, os.O_WRONLY | os.O_CREAT | os.O_APPEND, 0644)
            self.Size = os.fstat(self.Fileno).st_size

    def __call__(self, s):
        self.CheckRotate()
        s = "%s - %s\n" % (datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S.%f"), s)
        if self.Fileno < 0:
            raise ValueError("Bad file descriptor for log file")
        self.Size += len(s)
        os.write(self.Fileno, s)
        if self.Sync:
            os.fsync(self.Fileno)

    def __del__(self):
        if self.Fileno > 0:
            os.close(self.Fileno)


def SysLog(Ident, s):
    syslog.openlog(ident=Ident, facility=syslog.LOG_LOCAL0)
    syslog.syslog(syslog.LOG_CRIT, s)
    syslog.closelog()


Log = OutLog
def SetLogger(L, Threaded=False, Tag=False):
    global Log
    if Threaded:
        Log = ThreadedLog(L, Tag)
    else:
        Log = L


# ==============================================================================================================


class Exit(Exception):
    pass


class TimedOut(Exception):
    pass


class TThreadPG(object):
    def __init__(self):
        sys.setcheckinterval(10)
        self.Dict = {}

    def Add(self, Pg):
        if Pg <= 0:
            return
        ThreadIdent = threading.current_thread().ident
        if ThreadIdent not in self.Dict:
            self.Dict[ThreadIdent] = set()
        self.Dict[ThreadIdent].add(Pg)

    def Del(self, Pg):
        ThreadIdent = threading.current_thread().ident
        self.Dict.get(ThreadIdent, set()).discard(Pg)

    def KillAll(self, ThreadIdent=None):
        if ThreadIdent is None:
            ThreadIdent = threading.current_thread().ident
        PgSet = self.Dict.get(ThreadIdent, set())
        while len(PgSet) > 0:
            try:
                Pg = PgSet.pop()
                os.killpg(Pg, 9)
            except (OSError, AttributeError):
                pass


class TThreadPGEmpty(object):
    def __init__(self):
        pass

    def Add(self, Pg):
        pass

    def Del(self, Pg):
        pass

    def KillAll(self, ThreadIdent=None):
        pass


ThreadPG = TThreadPGEmpty()
def SetThreadExec():
    global ThreadPG
    ThreadPG = TThreadPG()


def Spawn(cmd, env={}):
    return os.spawnve(os.P_NOWAIT, cmd[0], cmd, env)


class Exec(object):
    def __init__(self, command, env={"LANG": "C"}):
        self.pid = -1
        try:
            self.p = subprocess.Popen(
                command,
                env=env,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                close_fds=True,
                preexec_fn=lambda: os.setpgid(os.getpid(), os.getpid())
            )
            self.pid = self.p.pid
            ThreadPG.Add(self.pid)
            self.out = self.p.stdout.read().strip()
            self.err = self.p.stderr.read().strip()
            self.p.wait()
            self.status = self.p.returncode
        except TimedOut:
            raise
        except OSError as e:
            Log("Got exception running %s: %s" % (command, e))
            raise
        finally:
            ThreadPG.Del(self.pid)
            self.cleanup()

    def cleanup(self):
        try:
            os.killpg(self.p.pid, 9)
        except (OSError, AttributeError):
            pass
        try:
            self.p.wait()
        except AttributeError:
            pass


class ExecPipe(object):
    def __init__(self, command, stdin=False, env={"LANG": "C"}):
        self.pid = -1
        try:
            self.p = subprocess.Popen(
                command,
                env=env,
                stdin=(subprocess.PIPE if stdin else None),
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                close_fds=True,
                preexec_fn=lambda: os.setpgid(os.getpid(), os.getpid())
            )
            self.pid = self.p.pid
            ThreadPG.Add(self.pid)
            self.stdin  = self.p.stdin
            self.stdout = self.p.stdout
            self.stderr = self.p.stderr
        except TimedOut:
            self.cleanup()
            raise
        except OSError as e:
            Log("Got exception running %s: %s" % (command, e))
            raise

    def wait(self):
        try:
            while self.p.poll() is None:
                time.sleep(0.05)
            self.status = self.p.returncode
        except TimedOut:
            self.cleanup()
            raise

    def cleanup(self):
        try:
            os.killpg(self.p.pid, 9)
        except (OSError, AttributeError):
            pass
        try:
            self.p.wait()
        except AttributeError:
            pass

    def __del__(self):
        ThreadPG.Del(self.pid)
        self.cleanup()


class ExecYield(object):
    def __init__(self, command, env={"LANG": "C"}):
        self.pid = -1
        try:
            self.p = subprocess.Popen(
                command,
                env=env,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                close_fds=True,
                preexec_fn=lambda: os.setpgid(os.getpid(), os.getpid())
            )
            self.pid = self.p.pid
            ThreadPG.Add(self.pid)
        except TimedOut:
            self.cleanup()
            raise
        except OSError as e:
            Log("Got exception running %s: %s" % (command, e))
            raise

    def read(self):
        try:
            while self.p.poll() is None:
                yield self.p.stdout.readline().strip('\n')
            self.status = self.p.returncode
        except TimedOut:
            self.cleanup()
            raise

    def cleanup(self):
        try:
            os.killpg(self.p.pid, 9)
        except (OSError, AttributeError):
            pass
        try:
            self.p.wait()
        except AttributeError:
            pass

    def __del__(self):
        ThreadPG.Del(self.pid)
        self.cleanup()


def EE(Cmd):
    e = Exec(Cmd)
    if e.status != 0:
        Log("Error in CMD (%s) OUT: %s ERR: %s" % (Cmd, e.out, e.err))
    else:
        Log("%s: %s" % (Cmd, repr(e.out)))
    return e.status


# ==============================================================================================================


def GetCommandFromPid(Pid):
    try:
        Line = GetFileData(os.path.join("/proc", str(Pid), "cmdline"))
        if len(Line) > 0:
            return Line.split('\x00')[0].split('/')[-1]
    except (ValueError, IndexError, KeyError):
        pass
    return None


def GetPidStartTime(Pid):
    try:
        Line = GetFileData(os.path.join("/proc", str(Pid), "stat"))
        if len(Line) > 0:
            return int(Line.split(None, 22)[21])
    except (ValueError, IndexError, KeyError):
        pass
    return None


def PGrep(ProcessName, ProcessFilter=None, Count=False):
    if ProcessFilter is None:
        return Exec(["/usr/bin/pgrep", "-x" + ("c" if Count else "o"), ProcessName.__str__()]).out
    return Exec(["/usr/bin/pgrep", "-f" + ("c" if Count else "o"), "/%s .*%s" % (ProcessName, ProcessFilter)]).out


def GetChildren(Pid, Recursive=False):
    PidList = Exec(["/usr/bin/pgrep", "-P", str(Pid)]).out.split()
    if Recursive:
        SubPidList = []
        for Pid in PidList:
            SubPidList += GetChildren(Pid, Recursive=True)
        PidList += SubPidList
    return PidList


def PID(Pid):
    try:
        Pid = int(Pid)
        return None if Pid == 0 else Pid
    except ValueError:
        return None


def PIDUptime(Pid=os.getpid()):
    for Line in GetFileData("/proc/stat").split('\n'):
        if Line.startswith("btime"):
            BootTime = int(Line.split()[1])
    Tick = os.sysconf(os.sysconf_names["SC_CLK_TCK"])
    StartTime = int(GetFileData("/proc/%d/stat" % Pid).split()[21])/Tick
    return BootTime + StartTime


# ==============================================================================================================


def GetFileData(FileName):
    Data = ""
    try:
        File = open(FileName, 'r')
        Data = File.read()
        File.close()
    except (IOError, OSError) as e:
        if e.errno == 2 or e.errno == 3:
            pass
        else:
            raise
    return Data


def PutFileData(FileName, Data):
    try:
        File = os.open(FileName, os.O_WRONLY | os.O_CREAT, 0644)
        os.write(File, Data)
        os.close(File)
    except OSError as e:
        if e.errno == 3:
            pass
        else:
            raise


def WriteFile(Name, Content, Mode='a'):
    with open(Name, Mode) as File:
        File.write(Content)


def CopyDirectoryTree(DirSrc, DirDst):
    DirSrc = os.path.realpath(DirSrc)
    DirDst = os.path.realpath(DirDst)
    if not os.path.exists(DirDst):
        os.makedirs(DirDst)
    for Item in os.listdir(DirSrc):
        Exec(["/bin/cp", "-a", os.path.join(DirSrc, Item), DirDst])


def Du(Path):
    S = 0
    for Root, Dirs, Files in os.walk(Path, topdown=False):
        for Name in Files:
            S += os.stat(os.path.join(Root, Name)).st_size
        for Name in Dirs:
            S += os.stat(os.path.join(Root, Name)).st_size
    return S


class LockFile(object):
    def __init__(self, FileName):
        self.FileName = FileName
        self.File = None

    def Lock(self, NonBlock=True):
        try:
            self.File = open(self.FileName, 'a')
            fcntl.lockf(self.File, fcntl.LOCK_EX | (fcntl.LOCK_NB if NonBlock else 0))
        except Exception as e:
            Log("Cannot lock %s: %s" % (self.FileName, e))
            return False
        return True

    def UnLock(self):
        try:
            self.File.close()
        except Exception as e:
            return False
        return True


# ==============================================================================================================


def GetHostname():
    return socket.gethostname()


def ResolveNameToIPv6(FQDN):
    try:
        return [i[-1] for i in socket.getaddrinfo(FQDN, 0) if i[0] == socket.AF_INET6][0][0]
    except (IndexError, socket.gaierror) as e:
        Log("Cannot resolve '%s': %s" % (FQDN, e))


def ResolveIPv6ToName(IPv6):
    try:
        return socket.gethostbyaddr(IPv6)[0]
    except (IndexError, socket.herror) as e:
        Log("Cannot resolve '%s': %s" % (IPv6, e))


def isIPv4(addr):
    try:
        socket.inet_pton(socket.AF_INET, addr)
    except socket.error:
        return False
    return True


def isIPv6(addr):
    try:
        socket.inet_pton(socket.AF_INET6, addr)
    except socket.error:
        return False
    return True


def IPv6ToNum(addr):
    hi, lo = struct.unpack('!QQ', socket.inet_pton(socket.AF_INET6, addr))
    return (hi<<64) | lo


def NumToIPv6(n):
    return socket.inet_ntop(socket.AF_INET6, struct.pack("!QQ", n>>64, n - ((n>>64)<<64)))


def IPv6FromPrefixPostfix(IPv6Prefix, IPv6Postfix):
    IPv6PrefixNum = IPv6ToNum(IPv6Prefix)
    Shift = 48 if (IPv6PrefixNum>>24) & 0xffff == 0xfffe else 32
    return NumToIPv6(((IPv6PrefixNum>>Shift)<<Shift) + IPv6ToNum("::" + IPv6Postfix))


def IPv6FromIPv6AndMAC(IPv6Prefix, MAC):
    IPv6PrefixNum = IPv6ToNum(IPv6Prefix)
    MACList = MAC.split(':')
    if (IPv6PrefixNum>>24) & 0xffff == 0xfffe:
        Shift = 64
        MACList = ["%0x" % (int(MACList[0], 16) ^ 0x02)] + MACList[1:3] + ["ff", "fe"] + MACList[3:]
    else:
        Shift = 32
        MACList = MACList[-4:]
    IPv6Postfix = ":".join(map("".join, zip(*[iter(MACList)]*2)))
    return NumToIPv6(((IPv6PrefixNum>>Shift)<<Shift) + IPv6ToNum("::" + IPv6Postfix))


def StringToMAC(String):
    md5cut = hashlib.md5(String).hexdigest()[:24:2]
    return ":".join(map("".join, zip(*[iter(md5cut)]*2)))


def GetIfaceForIP(IP):
    IPv4OnIfacesDict = {}
    IPv6OnIfacesDict = {}
    for AddrLine in Exec("/sbin/ip -o addr list scope global".split()).out.split('\n'):
        AddrLineArray = AddrLine.split()
        IPAddress = AddrLineArray[3].split('/')[0]
        Iface = AddrLineArray[1]
        if isIPv4(IPAddress):
            IPv4OnIfacesDict[IPAddress] = Iface
        else:
            IPv6OnIfacesDict[IPAddress] = Iface
    for AddrLine in Exec("/sbin/ip -o -5 tunnel show".split()).out.split('\n'):
        if len(AddrLine) > 0:
            AddrLineArray = AddrLine.split()
            Iface = AddrLineArray[0][:-1]
            IPv6OnIfacesDict[AddrLineArray[3]] = Iface
    return IPv4OnIfacesDict[IP] if IP in IPv4OnIfacesDict else IPv6OnIfacesDict[IP] if IP in IPv6OnIfacesDict else None


def GetURL(URL, Headers={}, Data=None, Timeout=4):
    try:
        Req  = urllib2.Request(URL, Data, Headers)
        Sock = urllib2.urlopen(Req, timeout=Timeout)
        for Line in Sock.read().split('\n'):
            yield Line
    except TimedOut:
        raise
    except Exception as e:
        Log("Exception in GetURL for '%s' - %s" % (URL, e.__str__()))
