import gevent
import gevent.lock
import gevent.queue
import thread
import threading
import Queue as tQ
import ctypes
import bisect
import time
import traceback
import sys

import system


# ==============================================================================================================


def Cut(Obj, Len=128):
    Obj = repr(Obj)
    if len(Obj) > Len:
        return Obj[:Len - 3] + "...'"
    return Obj


# ==============================================================================================================


def ExcThread(ThreadIdent, Exc):
    system.ThreadPG.KillAll(ThreadIdent)
    Exc = ctypes.py_object(Exc)
    Res = ctypes.pythonapi.PyThreadState_SetAsyncExc(ctypes.c_long(ThreadIdent), Exc)
    if Res > 1:
        ctypes.pythonapi.PyThreadState_SetAsyncExc(ThreadIdent, None)
        print("PyThreadState_SetAsyncExc failed")


def TraceTime(StopTime):
    def CallTrace(Frame, Event, Arg):
        def LineTrace(Frame, Event, Arg):
            if time.time() >= StopTime:
                raise system.TimedOut
            return LineTrace
        if time.time() >= StopTime:
            raise system.TimedOut
        return LineTrace if Event == "call" and Frame.f_globals["__name__"] == "subprocess" else None
    sys.settrace(CallTrace)


def TraceTimeFull(StopTime):
    def CallTrace(Frame, Event, Arg):
        def LineTrace(Frame, Event, Arg):
            if time.time() >= StopTime:
                raise system.TimedOut
            return LineTrace
        return LineTrace if Event == "call" else None
    sys.settrace(CallTrace)


class TraceTimeout(object):
    def __init__(self, Timeout):
        self.StopTime = time.time() + Timeout

    def __enter__(self):
        TraceTimeFull(self.StopTime)

    def __exit__(self, Type, Value, TB):
        sys.settrace(None)


class TTimeOutSetter(object):
    def __init__(self, TimeOut, Id, Time, Trace):
        self.TimeOut = TimeOut
        self.Id      = Id
        self.Time    = Time
        self.Trace   = Trace
        self.Ident   = thread.get_ident()

    def __enter__(self):
        if self.Trace:
            TraceTime(self.Time)
        self.TimeOut.Arm(self.Id, self.Ident, self.Time)

    def __exit__(self, Type, Value, TB):
        if self.Trace:
            sys.settrace(None)
        self.TimeOut.Disarm(self.Id, self.Ident)


class TTimeOuter(object):
    STOP   = 0
    ARM    = 1
    DISARM = 2

    def __init__(self):
        self.IdCounter  = 0
        self.Queue      = tQ.Queue()
        self.MainThread = threading.Thread(target=self.Runner)
        self.MainThread.daemon = True
        self.MainThread.start()

    def Runner(self):
        Timeout = None
        QueueList = []
        while True:
            try:
                Action, Id, ThreadIdent, Time = self.Queue.get(True, Timeout)
                if Action == self.DISARM:
                    for i, Item in enumerate(QueueList):
                        if Item[1] == Id:
                            del QueueList[i]
                            break
                elif Action == self.ARM:
                    bisect.insort_left(QueueList, (Time, Id, ThreadIdent))
                elif Action == self.STOP:
                    while QueueList:
                        ExcThread(QueueList.pop()[2], system.TimedOut)
                        self.Queue.task_done()
                    return
                self.Queue.task_done()
            except tQ.Empty:
                pass
            while len(QueueList) > 0:
                if QueueList[0][0] <= time.time():
                    ExcThread(QueueList[0][2], system.TimedOut)
                    del QueueList[0]
                else:
                    Timeout = QueueList[0][0] - time.time()
                    break
            else:
                Timeout = None
                continue

    def Arm(self, Id, ThreadIdent, Time):
        self.Queue.put((self.ARM, Id, ThreadIdent, Time))

    def Disarm(self, Id, ThreadIdent):
        self.Queue.put((self.DISARM, Id, ThreadIdent, 0))

    def Set(self, Timeout, Trace=False):
        self.IdCounter = (self.IdCounter + 1)%(1<<24)
        return TTimeOutSetter(self, self.IdCounter, time.time() + Timeout, Trace)

    def Stop(self):
        self.Queue.put((self.STOP, 0, 0, 0))
        self.MainThread.join()


# ==============================================================================================================


class TSemaphore(object):
    def __init__(self, Semaphore):
        self.Semaphore = Semaphore()

    def acquire(self):
        return self.Semaphore.acquire()

    def release(self):
        return self.Semaphore.release()

    def __enter__(self):
        # print("Got Lock in %s" % threading.current_thread().name)
        self.Semaphore.acquire()

    def __exit__(self, Type, Value, TB):
        # print("Release Lock in %s" % threading.current_thread().name)
        self.Semaphore.release()


# ==============================================================================================================


def ThreadsSpawn(Name, Function, *Args, **Kwargs):
    T = threading.Thread(target=Function, name=Name, args=Args, kwargs=Kwargs)
    T.daemon = True
    T.start()
    return T


def GeventSpawn(Name, Function, *Args, **Kwargs):
    return gevent.spawn(Function, *Args, **Kwargs)


def ThreadsJoinAll(TaskList):
    while TaskList:
        DeadList = []
        for T in TaskList:
            if T.isAlive():
                T.join(0.05)
            else:
                DeadList.append(T)
                yield T
        TaskList = [T for T in TaskList if T not in DeadList]


def GeventJoinAll(TaskList):
    for T in gevent.joinall(TaskList):
        yield T


def ThreadsKillAll(TaskList):
    for T in TaskList:
        while T.isAlive():
            ExcThread(T.ident, system.Exit)
            time.sleep(0.05)


def GeventKillAll(TaskList):
    gevent.killall(TaskList)


def ThreadsTimeoutSet(Timeout):
    return TimeOuter.Set(Timeout) if TimeOuter else TraceTimeout(Timeout)


def GeventTimeoutSet(Timeout):
    return gevent.Timeout(Timeout, system.TimedOut)


def ThreadsSleep(Time):
    Time += time.time()
    while Time > time.time():
        time.sleep(0.005)


def GeventSleep(Time):
    return gevent.sleep(Time)


def ThreadsLock():
    return TSemaphore(threading.Semaphore)


def GeventLock():
    return TSemaphore(gevent.lock.Semaphore)


def ThreadsQueue():
    return tQ.Queue()


def GeventQueue():
    return gevent.queue.Queue()


# ==============================================================================================================


def UseGevent():
    global Spawn
    global JoinAll
    global KillAll
    global TimeoutSet
    global Sleep
    global Lock
    global Queue
    Spawn       = GeventSpawn
    JoinAll     = GeventJoinAll
    KillAll     = GeventKillAll
    TimeoutSet  = GeventTimeoutSet
    Sleep       = GeventSleep
    Lock        = GeventLock
    Queue       = GeventQueue


def UseThreads(ExcTimeOuter=False):
    global Spawn
    global JoinAll
    global KillAll
    global TimeoutSet
    global Sleep
    global Lock
    global Queue
    global TimeOuter
    Spawn       = ThreadsSpawn
    JoinAll     = ThreadsJoinAll
    KillAll     = ThreadsKillAll
    TimeoutSet  = ThreadsTimeoutSet
    Sleep       = ThreadsSleep
    Lock        = ThreadsLock
    Queue       = ThreadsQueue
    TimeOuter   = TTimeOuter() if ExcTimeOuter else None
    if ExcTimeOuter:
        system.SetThreadExec()


# ==============================================================================================================


class TTasksLoop(object):
    def __init__(self, Threaded=False, ExcTimeOuter=False):
        if Threaded:
            UseThreads(ExcTimeOuter)
            self.Log("Taskloop is using threads backend for tasks")
        else:
            UseGevent()
            self.Log("Taskloop is using gevent backend for tasks")
        self.TaskDict   = {}

    def Log(self, String):
        system.Log(String)

    def StartTask(self, Function, Args=(), Kwargs={}, Period=60, Timeout=10, MinDistance=0, Jitter=0, Name=None, Tag=None):
        self.TaskDict[Spawn(
            Name if Name else Function.__name__,
            self.Task,
            Function,
            Args,
            Kwargs,
            Period,
            Timeout,
            MinDistance,
            Jitter,
            Tag
        )] = {
            "function": Function,
            "args":     Args,
            "kwargs":  	Kwargs,
            "period":   Period,
            "timeout":  Timeout,
            "distance": MinDistance,
            "jitter":   Jitter,
            "tag":      Tag
        }

    def Task(self, Function, Args, Kwargs, Period, Timeout, MinDistance, Jitter, Tag):
        def Log(S):
            self.Log("%s%s" % (TagText, S))

        TagText = "[%s] " % Tag if Tag is not None else ""
        FormattedArgs = ", ".join(["%s" % A for A in Args] + ["%s=%s" % (K, V) for K, V in Kwargs.items()])
        GlobalException = None
        while True:
            try:
                if GlobalException is not None:
                    Log("global exception for %s='%s'" % (Function.__name__, GlobalException))
                    Sleep(10)
                    GlobalException = None
                TimeStart = time.time()
                TimeJitter = system.Random(Jitter) - Jitter/2.0
                Log("running %s(%s) period: %.3f timeout: %.3f distance: %.3f jitter %.3f (%.3f)" % (
                    Function.__name__,
                    FormattedArgs,
                    Period,
                    Timeout,
                    MinDistance,
                    Jitter,
                    TimeJitter))
                try:
                    r = None
                    if Timeout <= 0:
                        r = Function(*Args, **Kwargs)
                    else:
                        with TimeoutSet(Timeout):
                            r = Function(*Args, **Kwargs)
                    Reason = "exited" if r is None else "exited return=%s" % Cut(r)
                except system.TimedOut:
                    Reason = "timed out"
                except system.Exit:
                    Reason = "exit request"
                    Period = -1
                except Exception:
                    EType, EValue, ETraceback = sys.exc_info()
                    Reason = "exception='\n%s%s'" % ("".join(traceback.format_tb(ETraceback)[1:]), EValue)
                except:
                    Log("general exception in task")
                    raise system.Exit
                DeltaTime = time.time() - TimeStart
                Log("stopped %s(%s) time: %.3f reason: %s" % (Function.__name__, FormattedArgs, DeltaTime, Reason))
                if Period <= 0:
                    break
                SleepTime = max(Period - DeltaTime, MinDistance) + TimeJitter
                if SleepTime > 0:
                    Sleep(SleepTime)
            except system.Exit:
                Log("exited %s(%s) reason: external exit call" % (Function.__name__, FormattedArgs))
                return
            except Exception as e:
                GlobalException = "".join(traceback.format_exception(*sys.exc_info()))

    def WaitAll(self):
        try:
            while True:
                TaskList = [G for G in self.TaskDict]
                if len(TaskList) == 0:
                    break
                for G in JoinAll(TaskList):
                    del self.TaskDict[G]
        except KeyboardInterrupt:
            self.Log("Taskloop is exiting ...")
            self.KillAll()

    def KillAll(self):
        KillAll([G for G in self.TaskDict])
