#!/usr/bin/env python

import os
import sys
import time
import json
import argparse

sys.path.append("/Berkanavt/solomon-backup/bin")

import db
import system


# ==============================================================================================================


YT_SECRET_PATH  = "/Berkanavt/keys/solomon/yt.secret"

HOME            = "/Berkanavt/solomon-backup"
CONFIGFILE      = os.path.join(HOME, "config/backup.conf")
LOGSDIR         = os.path.join(HOME, "logs")

GROUPFILE       = "/etc/yandex/group"
GROUPHOSTSFILE  = "/etc/yandex/group_hosts"


# ==============================================================================================================


def MetricsAreMine(Force):
    global MyYDB
    global MyTablesList
    global MyYtToken
    global YtProxy
    global YtPath

    RETRYCOUNT  = 10
    for _1 in range(RETRYCOUNT):
        try:
            CfgFile = open(CONFIGFILE)
            Cfg = json.load(CfgFile)
            CfgFile.close()
            break
        except (ValueError, IOError) as e:
            system.Log("Failed to read config %s: %s" % (CONFIGFILE, e))
        time.sleep(1)
        system.Log("Retrying configuration file read ...")
    else:
        system.Log("Failed to read configuration file.")
        return False

    for _1 in range(RETRYCOUNT):
        MyCluster   = system.GetFileData(GROUPFILE).strip()
        ServersList = system.GetFileData(GROUPHOSTSFILE).strip().split()
        if len(MyCluster) > 0 and len(ServersList) > 0:
            break
        system.Log("Failed to read group files %s or %s" % (GROUPFILE, GROUPHOSTSFILE))
        time.sleep(1)
        system.Log("Retrying configuration files read ...")
    else:
        system.Log("Failed to read configuration files.")
        return False
    system.Log("Successfully read configuration files")

    MyYDB       = Cfg[MyCluster].get("ydb", None)
    YdbPathList = Cfg[MyCluster].get("metapaths", [])
    YdbSkipList = Cfg[MyCluster].get("skippaths", [])
    KVPath      = Cfg[MyCluster].get("kvpath", None)

    if not KVPath:
        return False

    if not Force:
        KVPath = os.path.join(MyYDB, KVPath)
        for _1 in range(RETRYCOUNT):
            KVList = db.KVLs(KVPath)
            if len(KVList) > 0:
                break
            time.sleep(1)
            system.Log("Retrying getting kv list ...")
        else:
            system.Log("Failed to get kv list.")
            return False
        Hostname = system.GetHostname()
        MyShardsList = [int(KV["Shard"]) for KV in KVList if KV["Host"] == Hostname]
        if 1 not in MyShardsList:
            return False

    for _1 in range(RETRYCOUNT):
        try:
            Secret = system.GetFileData(YT_SECRET_PATH).strip()
            if len(Secret) > 0:
                break
        except (ValueError, IOError) as e:
            system.Log("Failed to read yt secret %s: %s" % (YT_SECRET_PATH, e))
        time.sleep(1)
        system.Log("Retrying yt secret file read ...")
    else:
        system.Log("Failed to read yt secret file.")
        return False

    os.environ["YT_TOKEN"] = Secret
    MyYtToken = Secret

    for _1 in range(RETRYCOUNT):
        TablesList = sorted([str(P) for SP in YdbPathList for P in db.YdbTree(MyYDB, os.path.join(MyYDB, SP), YdbSkipList)])
        if len(TablesList) > 0:
            break
        time.sleep(1)
        system.Log("Retrying getting ydb tables ...")
    else:
        system.Log("Failed to get list of ydb tables.")
        return False

    MyTablesList = [T for T in TablesList if T[-11:-2] == "/Metrics_"]

    if MyCluster == "solomon_prod_storage_vla":
        YtProxy = "arnold"
        YtPath = "//home/solomon/VLA_PROD"
    elif MyCluster == "solomon_prod_storage_sas":
        YtProxy = "hahn"
        YtPath = "//home/solomon/SAS_PROD"
    else:
        return False

    return True


# ==============================================================================================================


def Fatal(Str):
    system.Log(Str)
    exit(1)


# ==============================================================================================================


def main():
    Parser = argparse.ArgumentParser(description="cluster manage script")
    Parser.add_argument("-v", "--verbose", action="store_true", help="log to stdout")
    Parser.add_argument("-f", "--force", action="store_true", help="ignore shard #1 as a marker to start backup")
    Parser.add_argument("--skip-transfer", action="store_true", help="skip transfer ydb->yt")
    Args = Parser.parse_args()

    if not Args.verbose:
        system.SetLogger(system.FileLog(os.path.join(LOGSDIR, "yt-backup.log"), 50*1024*1024, 10, True))
    else:
        system.SetLogger(system.OutLog)
    system.Log("==============================================================================================================")

    if db.NO_YT:
        system.Log("Will not backup Metrics_* to YT: no yt-python package found")
        exit(0)

    if not MetricsAreMine(Args.force):
        system.Log("It is not my duty to backup Metrics_* to YT")
        exit(0)

    YtTmpTable    = "%s/Metrics_tmp" % YtPath
    YtResultTable = "%s/Metrics" % YtPath
    YtDstTables   = ["%s/%s" % (YtPath, T.split('/')[-1]) for T in MyTablesList]

    RETRYCOUNT = 5
    Yt = db.TYt(YtProxy, MyYtToken)

    with Yt.Transaction():
        if not Yt.Lock(YtResultTable):
            Fatal("Failed to lock %s" % YtResultTable)
        try:
            system.Log("Checking previous ydb->yt transfers")
            if not db.YdbExportsCleanup(MyYDB):
                Fatal("Failed to cleanup exports")
            system.Log("Sleeping for 15 seconds to allow ydb to settle down after the exports cleanup")
            time.sleep(15)
            ExportsList = db.YdbExportsList(MyYDB)
            for E in ExportsList:
                system.Log("Following export is active after the cleanup: id=%s status=%s ready=%s progress=%s" % (E["id"], E["status"], E["ready"], E["progress"]))
                FromSet = set(P for P in E["items"]) & set(MyTablesList)
                ToSet   = set(P for _, P in E["items"].items()) & set(YtDstTables)
                if len(FromSet) > 0:
                    Fatal("Yt export from my src tables is running (id=%s): %s" % (E["id"], list(FromSet)))
                if len(ToSet) > 0:
                    Fatal("Yt export to my dst tables is running (id=%s): %s" % (E["id"], list(ToSet)))
            if not Args.skip_transfer:
                system.Log("Checking old ydb tables in yt")
                for DstTable in YtDstTables + [YtTmpTable]:
                    if Yt.Exist(DstTable):
                        if not Yt.Remove(DstTable):
                            Fatal("Failed to remove old table from yt: %s ." % DstTable)
                        system.Log("Removed old table from yt: %s" % DstTable)
        finally:
            Yt.UnLock(YtResultTable)
            system.Log("Finishing transaction to flush all removed tables in yt")

    if not Args.skip_transfer:
        with Yt.Transaction():
            if not Yt.Lock(YtResultTable):
                Fatal("Failed to lock %s" % YtResultTable)
            try:
                system.Log("Starting ydb->yt transfer")
                TransferId = db.YdbExportToYt(MyYDB, {T: "{}/{}".format(YtPath, T.split('/')[-1]) for T in MyTablesList}, YtProxy)
                if not TransferId:
                    Fatal("Cannot start yt export of %s to %s ." % (MyTablesList, YtPath))
                if not db.YdbOperationWait(MyYDB, TransferId):
                    Fatal("Failed yt export of %s to %s ." % (MyTablesList, YtPath))
            finally:
                Yt.UnLock(YtResultTable)
                system.Log("Finishing transaction to get all tables in yt")

    for _1 in range(RETRYCOUNT):
        system.Log("Checking if all tables are present in yt")
        ExistsDict = {DstTable: Yt.Exist(DstTable) for DstTable in YtDstTables}
        if False not in set(V for _, V in ExistsDict.items()):
            break
        time.sleep(5)
        system.Log("Missing tables in yt: %s" % [T for T, V in ExistsDict.items() if not V])
    else:
        Fatal("Failed to get tables in yt.")

    with Yt.Transaction():
        if not Yt.Lock(YtResultTable):
            Fatal("Failed to lock %s" % YtResultTable)
        try:
            system.Log("Merging yt tables: %s" % YtDstTables)
            if not Yt.Sort(YtDstTables, YtTmpTable, ["shardId", "hash", "labels"]):
                Yt.Remove(YtTmpTable)
                Fatal("Failed to merge %s to %s ." % (YtDstTables, YtTmpTable))
            system.Log("Renaming tmp table: %s" % YtTmpTable)
            if not Yt.Rename(YtTmpTable, YtResultTable):
                Fatal("Failed to rename yt tmp table %s to %s ." % (YtTmpTable, YtResultTable))
            system.Log("Removing transfer tables in yt: %s" % YtDstTables)
            if not Yt.Remove(YtDstTables):
                Fatal("Failed to remove %s ." % YtDstTables)
        finally:
            Yt.UnLock(YtResultTable)

    system.Log("Done.")


# ==============================================================================================================


if __name__ == "__main__":
    main()
