#!/usr/bin/env python

import os
import sys
import time
import json
import argparse

sys.path.append("/Berkanavt/solomon-backup/bin")

import db
import system


# ==============================================================================================================


YT_SECRET_PATH  = "/Berkanavt/keys/solomon/yt.secret"

HOME            = "/Berkanavt/solomon-backup"
CONFIGFILE      = os.path.join(HOME, "config/backup.conf")
LOGSDIR         = os.path.join(HOME, "logs")

GROUPFILE       = "/etc/yandex/group"
GROUPHOSTSFILE  = "/etc/yandex/group_hosts"


# ==============================================================================================================


def MetaIsMine(Force):
    global MyYDB
    global MyTablesList
    global MyYtToken
    global YtProxy
    global YtPath

    RETRYCOUNT  = 10
    for _1 in range(RETRYCOUNT):
        try:
            CfgFile = open(CONFIGFILE)
            Cfg = json.load(CfgFile)
            CfgFile.close()
            break
        except (ValueError, IOError) as e:
            system.Log("Failed to read config {}: {}".format(CONFIGFILE, e))
        time.sleep(1)
        system.Log("Retrying configuration file read ...")
    else:
        system.Log("Failed to read configuration file.")
        return False

    for _1 in range(RETRYCOUNT):
        MyCluster   = system.GetFileData(GROUPFILE).strip()
        ServersList = system.GetFileData(GROUPHOSTSFILE).strip().split()
        if len(MyCluster) > 0 and len(ServersList) > 0:
            break
        system.Log("Failed to read group files {} or {}".format(GROUPFILE, GROUPHOSTSFILE))
        time.sleep(1)
        system.Log("Retrying configuration files read ...")
    else:
        system.Log("Failed to read configuration files.")
        return False
    system.Log("Successfully read configuration files")

    if MyCluster != "solomon_prod_kfront":
        system.Log("Cannot backup meta on {}".format(MyCluster))
        return False

    MyYDB       = Cfg[MyCluster].get("ydb", None)
    YdbPathList = [
        "MegaGraphite",
        "Solomon",
    ]
    MySkipList = [
        "LastUpdates",
        "Locks",
    ]

    if not Force:
        if int(time.strftime("%j")) % len(ServersList) != ServersList.index(system.GetHostname()):
            system.Log("Daily backup of ydb tables now should be elsewhere.")
            return False

    for _1 in range(RETRYCOUNT):
        try:
            Secret = system.GetFileData(YT_SECRET_PATH).strip()
            if len(Secret) > 0:
                break
        except (ValueError, IOError) as e:
            system.Log("Failed to read yt secret {}: {}".format(YT_SECRET_PATH, e))
        time.sleep(1)
        system.Log("Retrying yt secret file read ...")
    else:
        system.Log("Failed to read yt secret file.")
        return False

    os.environ["YT_TOKEN"] = Secret
    MyYtToken = Secret

    for _1 in range(RETRYCOUNT):
        MyTablesList = sorted([str(P) for SP in YdbPathList for P in db.YdbTree(MyYDB, os.path.join(MyYDB, SP), MySkipList)])
        if len(MyTablesList) > 0:
            break
        time.sleep(1)
        system.Log("Retrying getting ydb tables ...")
    else:
        system.Log("Failed to get list of ydb tables.")
        return False

    YtProxy = "hahn"
    YtPath = "//home/solomon/META"

    return True


# ==============================================================================================================


def Fatal(Str):
    system.Log(Str)
    exit(1)


# ==============================================================================================================


def main():
    Parser = argparse.ArgumentParser(description="cluster manage script")
    Parser.add_argument("-v", "--verbose", action="store_true", help="log to stdout")
    Parser.add_argument("-f", "--force", action="store_true", help="ignore shard #1 as a marker to start backup")
    Parser.add_argument("--skip-transfer", action="store_true", help="skip transfer ydb->yt")
    Args = Parser.parse_args()

    if not Args.verbose:
        system.SetLogger(system.FileLog(os.path.join(LOGSDIR, "yt-meta-backup.log"), 50*1024*1024, 10, True))
    else:
        system.SetLogger(system.OutLog)
    system.Log("==============================================================================================================")

    if db.NO_YT:
        system.Log("Will not backup meta to YT: no yt-python package found")
        exit(0)

    if not MetaIsMine(Args.force):
        exit(0)

    TooOldTime = time.time() - 30*86400
    LastWeekTime = time.time() - 7*86400
    YtLockTable  = "{}/Lock".format(YtPath)
    YtBackupDict = {T: "{}/{}".format(YtPath, "/".join(T.split('/')[2:])) for T in MyTablesList}
    YtBackupTmpDict = {T: "{}_tmp".format(D) for T, D in YtBackupDict.items()}

    RETRYCOUNT = 5
    Yt = db.TYt(YtProxy, MyYtToken)

    with Yt.Transaction():
        if not Yt.Lock(YtLockTable):
            Fatal("Failed to lock {}".format(YtLockTable))
        try:
            system.Log("Checking previous ydb->yt transfers")
            if not db.YdbExportsCleanup(MyYDB):
                Fatal("Failed to cleanup exports")
            system.Log("Sleeping for 15 seconds to allow ydb to settle down after the exports cleanup")
            time.sleep(15)
            ExportsList = db.YdbExportsList(MyYDB)
            for E in ExportsList:
                system.Log("Following export is active after the cleanup: id={} status={} ready={} progress={}".format(
                    E["id"], E["status"], E["ready"], E["progress"]
                ))
                FromSet = set(E["items"].keys()) & set(YtBackupTmpDict.keys())
                ToSet   = set(E["items"].values()) & set(YtBackupTmpDict.values())
                if len(FromSet) > 0:
                    Fatal("Yt export from my src tables is running (id={}): {}".format(E["id"], list(FromSet)))
                if len(ToSet) > 0:
                    Fatal("Yt export to my dst tables is running (id={}): {}".format(E["id"], list(ToSet)))
            if not Args.skip_transfer:
                system.Log("Checking old ydb tables in yt")
                for DstTable in YtBackupTmpDict.values():
                    if Yt.Exist(DstTable):
                        if not Yt.Remove(DstTable):
                            Fatal("Failed to remove old table from yt: {} .".format(DstTable))
                        system.Log("Removed old table from yt: {}".format(DstTable))
        finally:
            Yt.UnLock(YtLockTable)
            system.Log("Finishing transaction to flush all removed tables in yt")

    if not Args.skip_transfer:
        with Yt.Transaction():
            if not Yt.Lock(YtLockTable):
                Fatal("Failed to lock {}".format(YtLockTable))
            try:
                system.Log("Starting ydb->yt transfer")
                TransferId = db.YdbExportToYt(MyYDB, YtBackupTmpDict, YtProxy)
                if not TransferId:
                    Fatal("Cannot start yt export of {} to {} .".format(YtBackupTmpDict.keys(), YtPath))
                if not db.YdbOperationWait(MyYDB, TransferId):
                    Fatal("Failed yt export of {} to {} .".format(YtBackupTmpDict.keys(), YtPath))
            finally:
                Yt.UnLock(YtLockTable)
                system.Log("Finishing transaction to get all tables in yt")

    for _1 in range(RETRYCOUNT):
        system.Log("Checking if all tables are present in yt")
        ExistsDict = {DstTable: Yt.Exist(DstTable) for DstTable in YtBackupTmpDict.values()}
        if False not in set(V for _, V in ExistsDict.items()):
            break
        time.sleep(5)
        system.Log("Missing tables in yt: {}".format([T for T, V in ExistsDict.items() if not V]))
    else:
        Fatal("Failed to get tables in yt.")

    with Yt.Transaction():
        if not Yt.Lock(YtLockTable):
            Fatal("Failed to lock {}".format(YtLockTable))
        try:
            system.Log("Renaming yt tmp tables: {}".format(YtBackupTmpDict.values()))
            for K, V in YtBackupTmpDict.items():
                if "MegaGraphite" not in K:
                    if not Yt.Rename(V, YtBackupDict[K]):
                        Fatal("Failed to rename yt tmp table {} -> {} .".format(V, YtBackupDict[K]))
            for K, V in YtBackupTmpDict.items():
                if "MegaGraphite" in K:
                    TargetName = YtBackupDict[K]
                    Path = V[:V.rfind("/")]
                    if not Path.startswith(YtPath):
                        Fatal("Failed to make directory from {}: {}".format(V, Path))
                    L = Yt.List(Path)
                    if not isinstance(L, list):
                        Fatal("Failed to list yt path {}: {}".format(Path, L))
                    try:
                        L = {
                            "{}/{}".format(Path, D["$value"]): time.mktime(time.strptime(D["$attributes"]["creation_time"], "%Y-%m-%dT%H:%M:%S.%fZ"))
                            for D in L if D["$attributes"]["type"] == "table"
                        }
                        system.Log("Got tables in {}: {}".format(Path, L))
                        TooOldTables = [D for D, C in L.items() if C < TooOldTime]
                        NoLastWeekTables = len([D for D, C in L.items() if C > LastWeekTime]) <= 2
                    except Exception as e:
                        Fatal("Exception while listing yt dir {}: {}".format(Path, e))
                    if len(TooOldTables) > 0 and not Yt.Remove(TooOldTables):
                        Fatal("Failed to remove too old tables from yt: {} .".format(TooOldTables))
                    if NoLastWeekTables and Yt.Exist(TargetName):
                        NewName = "{}.{}".format(TargetName, time.strftime("%Y-%m-%d"))
                        system.Log("No last week tables, renaming current table {} -> {}".format(TargetName, NewName))
                        if not Yt.Rename(TargetName, NewName):
                            Fatal("Failed to rename yt tmp table {} -> {} .".format(TargetName, NewName))
                    if not Yt.Rename(V, TargetName):
                        Fatal("Failed to rename yt tmp table {} -> {} .".format(V, TargetName))
        finally:
            Yt.UnLock(YtLockTable)

    system.Log("Done.")


# ==============================================================================================================


if __name__ == "__main__":
    main()
