#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os
import sys
import re
import socket
import xml.etree.ElementTree as ET
import datetime
import ssl
import urllib.request
import urllib.parse
import urllib.error
import gzip
import struct
import bisect


# ====================================================================================================================


DIR          = "/etc/solomon"
ENV_FILE     = os.path.join(DIR, "env")
DC_FILE      = os.path.join(DIR, "dc")
CLUSTER_FILE = os.path.join(DIR, "cluster")


DCNames      = {"Сасово": "sas", "Владимир": "vla", "Мянтсяля": "man", "Ивантеевка": "iva", "Мытищи": "myt"}
DCList       = ["sas", "vla", "man", "iva", "myt"]
ClusterList  = ["sts1", "sts2"]
EnvDict      = {
    "solomon_prod":             "production",
    "solomon_dev":              "development",
    "solomon_test":             "testing",
    "solomon_test_sts":         "prestable",
    "solomon_pre":              "prestable",
    "cloud_prod_solomon":       "cloud-prod",
    "cloud_preprod_solomon":    "cloud-preprod",
    "grafana_dev":              "testing",
    "grafana_prod":             "production",
    "graphite_bs":              "production",
    "graphite_ps":              "production",
    "graphite_gr":              "production",
    "mega_graphite_bs":         "production",
    "mega_graphite_ps":         "production",
    "mega_graphite_gr":         "production",
    "mega_graphite_test":       "testing"
}


# ====================================================================================================================


def Log(String):
    print("{} - {}".format(datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S.%f"), String))


def PutFileData(FileName, Data, IsTesting=False):
    if IsTesting:
        Log("Put data to file {}:\n{}\n".format(FileName, Data))
        return
    Dir = os.path.dirname(FileName)
    if not os.path.isdir(Dir):
        try:
            os.makedirs(os.path.dirname(FileName))
        except Exception as e:
            Log("Exception in PutFileData for {}: {}".format(FileName, e))
    try:
        File = os.open(FileName, os.O_WRONLY | os.O_CREAT, 0o644)
        os.write(File, Data.encode("utf-8"))
        os.close(File)
    except OSError as e:
        if e.errno == 3:
            pass
        else:
            raise


def IPv6ToNum(addr):
    hi, lo = struct.unpack('!QQ', socket.inet_pton(socket.AF_INET6, addr))
    return (hi << 64) | lo


def NumToIPv6(n):
    return socket.inet_ntop(socket.AF_INET6, struct.pack("!QQ", n >> 64, n & 0xffffffffffffffff))


def IPInNet(IPNum, NetPrefix, PrefixLength):
    Shift = 128 - PrefixLength
    return NetPrefix>>Shift == IPNum>>Shift


def GetHostname():
    return socket.gethostname()


def ResolveNameToIPv6(FQDN):
    try:
        return [i[-1] for i in socket.getaddrinfo(FQDN, 0) if i[0] == socket.AF_INET6][0][0]
    except (IndexError, socket.gaierror) as e:
        Log("Cannot resolve '{}': {}".format(FQDN, e))


class GetURL:
    def __init__(self, URL, Headers={}, Data=None, Timeout=10, Verify=False, Method=None, Decode=True):
        Ctx = ssl.create_default_context()
        if not Verify:
            Ctx.check_hostname = False
            Ctx.verify_mode    = ssl.CERT_NONE
        self.code   = 0
        self.err    = ""
        self.text   = ""
        self.data   = bytes()
        try:
            if isinstance(Data, dict):
                Data = urllib.parse.urlencode(Data)
            Req  = urllib.request.Request(URL, data=Data, headers=Headers, method=Method)
            Resp = urllib.request.urlopen(Req, timeout=Timeout, context=Ctx)
            self.code   = Resp.getcode()
            self.data   = Resp.read()
            if Decode:
                self.text = self.data.decode("utf-8")
        except urllib.error.HTTPError as e:
            self.code   = e.code
            self.err    = e.reason
            Log("HTTPError in GetURL for {} ({}) - {}".format(URL, self.code, self.err))
        except urllib.error.URLError as e:
            self.code   = 0
            Log("URLError in GetURL for {} - {}".format(URL, e))
        except Exception as e:
            Log("General exception in GetURL for {} - {}".format(URL, e))


# ====================================================================================================================


class IPDC:
    def __init__(self):
        def StringToNet(String):
            try:
                Net, PrefixLen = String.split('/')
                return IPv6ToNum(Net), int(PrefixLen)
            except (OSError, ValueError):
                return None, None

        def StringToLocation(String):
            for S, Name in DCNames.items():
                if String.startswith(S):
                    return Name
            return None

        self.NetToLocationDict = {}     # {{NetNum, PrefixNum}: Location, ... }
        self.NetPrefixesSorted = []     # [NetNum, ... ]
        self.NetPrefixesToPLen = {}     # {NetNum: {PrefixNum, ...}, ... }
        R = GetURL("https://ro.racktables.yandex.net/export/networklist-perdc.txt.gz", Decode=False)
        if R.code == 200:
            try:
                for Line in gzip.decompress(R.data).decode("utf-8").split('\n'):
                    Fields = Line.split('\t')
                    if len(Fields) != 4:
                        continue
                    Net, PrefixLen = StringToNet(Fields[0])
                    if not Net:
                        continue
                    self.NetToLocationDict[Net, PrefixLen] = StringToLocation(Fields[2])
                    if Net in self.NetPrefixesToPLen:
                        self.NetPrefixesToPLen[Net].add(PrefixLen)
                    else:
                        self.NetPrefixesToPLen[Net] = {PrefixLen}
                self.NetPrefixesSorted = sorted([P for P, _ in self.NetToLocationDict])
            except Exception as e:
                Log("Failed to get network list ({}): {}".format(R.code, e))

    def GetDc(self, IP):
        IPNum = IPv6ToNum(IP)
        Idx = bisect.bisect_left(self.NetPrefixesSorted, IPNum)
        while Idx > 0:
            Idx -= 1
            NetPrefix = self.NetPrefixesSorted[Idx]
            for L in sorted(self.NetPrefixesToPLen[NetPrefix], reverse=True):
                if IPInNet(IPNum, NetPrefix, L):
                    return self.NetToLocationDict[(NetPrefix, L)]
        return None


class TConductor:
    def __init__(self, Host):
        self.Host = Host
        self.Dict = {}
        R = GetURL("http://c.yandex-team.ru/api/hosts/{}".format(Host))
        if R.code == 200:
            try:
                XMLRoot = ET.fromstring(R.text)
                for Item in XMLRoot:
                    self.Dict[Item.find("fqdn").text] = {
                        "id":          Item.find("id").text,
                        "group":       Item.find("group").text,
                        "datacenter":  Item.find("datacenter").text,
                        "cluster":     Item.find("root_datacenter").text,
                        "short_name":  Item.find("short_name").text,
                        "description": Item.find("description").text
                    }
            except Exception as e:
                Log("Failed to get info for host '{}' - '{}': {}".format(Host, R.text, e))

    def Group(self):
        return self.Dict.get(self.Host, {}).get("group")

    def DC(self):
        return self.Dict.get(self.Host, {}).get("cluster")


# ====================================================================================================================


def HeuristicSearch(TargetList, SourceList):
    Re = re.compile("(^|[^a-zA-Z])(%s)([^a-zA-Z]|$)" % "|".join(SourceList))
    for Target in TargetList:
        if Target:
            Search = Re.search(Target)
            if Search:
                return Search.group(2)
    return None


# ====================================================================================================================


def main():
    IsTesting = False
    Hostname = GetHostname()
    if len(sys.argv) > 1 and sys.argv[1] == "-t":
        IsTesting = True
        if len(sys.argv) > 2:
            Hostname = sys.argv[2]

    CHost = TConductor(Hostname)
    Group = CHost.Group()
    DC    = CHost.DC()

    if DC not in DCList:
        DC = HeuristicSearch([Group, Hostname], DCList)
        if DC not in DCList:
            IP = ResolveNameToIPv6(Hostname)
            if IP:
                DC = IPDC().GetDc(IP)
            if DC not in DCList:
                Log("Cannot get DC from group ({}) or hostname ({})".format(Group, Hostname))
                exit(1)
    PutFileData(DC_FILE, DC, IsTesting)

    for G in EnvDict:
        if Group.startswith(G):
            PutFileData(ENV_FILE, EnvDict[G], IsTesting)
            break
    else:
        Log("Bad group: {}".format(Group))
        exit(1)

    ClusterResult = HeuristicSearch([Group, Hostname], ClusterList)
    if ClusterResult:
        PutFileData(CLUSTER_FILE, ClusterResult, IsTesting)


if __name__ == "__main__":
    main()
