#!/usr/bin/env python

import os
import time
import fcntl
from subprocess import call, Popen, PIPE


#=============================================================================================================


SERVER = "grpc://localhost:2135"
BACKUPDIR = "/Berkanavt/solomon/kikimr-backup"
LOCKFILE = os.path.join(BACKUPDIR, "lock")
TMPDIR = os.path.join(BACKUPDIR, "tmp")


#=============================================================================================================


def Log(s):
    print("%s - %s" % (time.strftime("%b %d %H:%M:%S"), s))


class Exec:
    def __init__(self, command):
        try:
            p = Popen(command, env={"LANG": "C"}, stdout=PIPE, stderr=PIPE, close_fds=True)
            self.out = p.stdout.read().strip()
            self.err = p.stderr.read().strip()
            p.wait()
            self.status = p.returncode
        except OSError as e:
            Log("Got exception running %s: %s" % (command, e))
            raise


def EE(Cmd):
    e = Exec(Cmd)
    if e.status != 0:
        Log("Error in CMD: %s OUT: %s ERR: %s" % (Cmd, e.out, e.err))
        exit(1)
    Log("%s: %s"  % (Cmd, repr(e.out)))


class LockFile(object):
    def __init__(self, FileName):
        self.FileName = FileName
        self.File = None

    def Lock(self, NonBlock=True):
        try:
            self.File = open(self.FileName, 'a')
            fcntl.lockf(self.File, fcntl.LOCK_EX | (fcntl.LOCK_NB if NonBlock else 0))
        except Exception as e:
            print("Cannot lock %s: %s" % (self.FileName, e))
            return False
        return True

    def UnLock(self):
        try:
            self.File.close()
        except Exception as e:
            return False
        return True


#=============================================================================================================


def KikimrLs(Path):
    e = Exec(["/usr/bin/kikimr", "-s", SERVER, "db", "schema", "ls", Path])
    return e.out


def CatToFile(KikimrPath, FsPath):
    Cmd = ["/usr/bin/kikimr", "-s", SERVER, "db", "readtable", "--format", "csv", KikimrPath]
    File = os.open(FsPath, os.O_CREAT | os.O_TRUNC | os.O_WRONLY, 0644)
    p = Popen(Cmd, env={"LANG": "C"}, stdout=File, stderr=PIPE, close_fds=True)
    p.wait()
    os.close(File)
    return p.returncode


def Tree(Path, SkipList = []):
    for DirEntry in KikimrLs(Path).split('\n'):
        if len(DirEntry) == 0:
            break
        _, Type, Name = DirEntry.split()
        FullName = os.path.join(Path, Name)
        if Type == "<dir>":
            Skip = False
            for SkipName in SkipList:
                if Name.startswith(SkipName):
                    Skip = True
                    break
            if not Skip:
                for DeepFullName in Tree(FullName, SkipList):
                    yield DeepFullName
        else:
            yield FullName


#=============================================================================================================


Lock = LockFile(LOCKFILE)
if not Lock.Lock():
    exit(0)

if len(TMPDIR) <= len(BACKUPDIR):
    Log("Bad TMPDIR: %s" % TMPDIR)
    exit(1)

EE(["/bin/rm", "-fr", TMPDIR])
EE(["/bin/mkdir", TMPDIR])

for KikimrFilePath in Tree("/", SkipList = ["MegaGraphite"]):
    FsFilePath = os.path.join(TMPDIR, KikimrFilePath.strip('/'))
    Exec(["/bin/mkdir", "-p", os.path.dirname(FsFilePath)])
    Log("copy %s -> %s" % (KikimrFilePath, FsFilePath))
    CatToFile(KikimrFilePath, FsFilePath)


BackupFileName = os.path.join(BACKUPDIR, time.strftime("kikimr-backup.%Y-%m-%d_%H.tgz"))
os.chdir(TMPDIR)
EE(["/bin/tar", "cvzf", BackupFileName, "."])
EE(["/bin/rm", "-r", TMPDIR])
EE(["/usr/bin/find", BACKUPDIR, "-type", "f", "-ctime", "+30", "-name", "kikimr-backup.*", "-delete"])
Lock.UnLock()
