#!/bin/sh


set -e


#-------------------------------------------------------------------------


install_conf() {
	echo "Installing configuration file: $1"
	[ -s "$1" -a ! -f "$1.BACKUP" ] && cp -f "$1" "$1.BACKUP"
	[ -f "$1.yandex" ] && cp -f "$1.yandex" "$1"
}

deinstall_conf() {
	echo "Deinstalling configuration file: $1"
	[ -s "$1" -a -f "$1.BACKUP" ] && cp -f "$1.BACKUP" "$1"
	rm -f "$1.BACKUP"
}

check_mod() {
	mod=$1 ; shift
	own=$1 ; shift

	echo "Checking file or directory $@ for mod $mod and owner $own"
	if [ -d "$@" -o -f "$@" ] ; then
		chmod $mod "$@"
		chown $own "$@"
	fi
}

check_exist() {
	what=$1 ; shift
	mod=$1 ; shift
	own=$1 ; shift

	case $what in
		-f)
			echo "Checking if file $@ exists"
			[ ! -f "$@" ] && mkdir -p "${@%/*}" && touch "$@"
			;;
		-d)
			echo "Checking if directory $@ exists"
			[ ! -d "$@" ] && mkdir -p "$@"
			;;
		*)
			;;
	esac
	check_mod $mod $own $@
}


#-------------------------------------------------------------------------


_install() {
	CRT="/etc/grafana/root.crt"
	NSSDB="/usr/share/grafana/.pki/nssdb"
	GRAFANACONF="/etc/grafana/grafana.ini"
	PASSFILE="/Berkanavt/keys/grafana/grafana.secrets"
	#
	# GRAFANA_PASSWORD : $PASS_GRAFANA
	# ADMIN_PASSWORD : $PASS_GRAFANA_WEB

	if [ -r "$PASSFILE" ] ; then
		eval `awk -F":" '{gsub(/[ \t]+/, ""); gsub(/[^A-Za-z0-9]/, "_", $1); print "PASS_" toupper($1) "=" $2}' $PASSFILE`
	else
		echo "CANNOT FIND $PASSFILE. Exiting..."
		exit
	fi
	if [ -z "$PASS_GRAFANA" -o -z "$PASS_GRAFANA_WEB" ] ; then
		echo "NO GRAFANA PASSWORDS. Exiting..."
		exit
	fi

	install_conf $GRAFANACONF
	sed -i "s/GRAFANA_PASSWORD/${PASS_GRAFANA}/
		s/ADMIN_PASSWORD/${PASS_GRAFANA_WEB}/" $GRAFANACONF
	check_mod 440 root:grafana $GRAFANACONF

	# Setup big dirs for grafana
	RUNDIR="/var/run/grafana"
	LIBDIR="/var/lib/grafana"
	LOGDIR="/var/log/grafana"
	LIBDIRBIG="/Berkanavt/grafana/data"
	LOGDIRBIG="/Berkanavt/grafana/logs"

	check_exist -d 755 grafana:grafana $RUNDIR
	check_mod 755 grafana:grafana $LIBDIRBIG
	check_mod 755 grafana:grafana $LOGDIRBIG

	if [ ! -L $LIBDIR ] ; then
		if [ -e $LIBDIR ] ; then
			echo "COPYING files..."
			tar cpf - -C $LIBDIR . | tar xvpf - -C $LIBDIRBIG
			echo "REMOVING files..."
			rm -vfr $LIBDIR
		fi
		ln -sf $LIBDIRBIG $LIBDIR
		echo "Need to restart grafana!"
	fi
	if [ ! -L $LOGDIR ] ; then
		if [ -e $LOGDIR ] ; then
			echo "COPYING files..."
			tar cpf - -C $LOGDIR . | tar xvpf - -C $LOGDIRBIG
			echo "REMOVING files..."
			rm -vfr $LOGDIR
		fi
		ln -sf $LOGDIRBIG $LOGDIR
		echo "Need to restart grafana!"
	fi

	if [ ! -d $NSSDB ] ; then
		mkdir -p $NSSDB
		certutil -d sql:$NSSDB -N --empty-password
		certutil -d sql:$NSSDB -A -n internal-root-ca -t C -i /etc/ssl/certs/YandexInternalRootCA.pem
		chown -R grafana:grafana $NSSDB
	fi

	# Fetch root CA
	wget "https://crls.yandex.net/allCAs.pem" -O $CRT
	check_mod 440 root:grafana $CRT
}


_rm() {
	echo "Deinstall"
}


#-------------------------------------------------------------------------


case "$1" in
	"")
		_install || true
	;;
	remove|purge|upgrade|disappear)
		_rm || true
	;;
esac
