#!/usr/bin/env python3

import re
import os
import time
import datetime
import random
import subprocess
import socket
import argparse


SecretFile = "/Berkanavt/keys/grafana/grafana.secrets"
SSLCAFile = "/etc/grafana/root.crt"
MysqlPassword = ""


# =============================================================================================================


def Log(s):
    TimeStr = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S.%f")
    print("{} - {}".format(TimeStr, s))


def GetFileData(FileName):
    Data = ""
    try:
        File = open(FileName, 'r')
        Data = File.read()
        File.close()
    except (IOError, OSError) as e:
        if e.errno == 2 or e.errno == 3:
            pass
        else:
            raise
    return Data


def Random(Number):
    return Number * random.random()


class Exec(object):
    def __init__(self, Command, Env={"LANG": "C"}, ErrToOut=False, Stdin=None, Stdout=True, Pipe=False):
        self.pid = -1
        self.out = ""
        self.err = ""
        self.status = 1
        try:
            self.p = subprocess.Popen(
                Command,
                env=Env,
                stdin=subprocess.PIPE if Stdin is True else Stdin,
                stdout=subprocess.PIPE if Stdout is True else Stdout,
                stderr=subprocess.STDOUT if ErrToOut else subprocess.PIPE,
                close_fds=True,
                preexec_fn=lambda: os.setpgid(os.getpid(), os.getpid())
            )
            self.pid = self.p.pid
            if Pipe:
                self.stdin = self.p.stdin
                self.stdout = self.p.stdout
                self.stderr = self.p.stderr
            else:
                try:
                    self.out = self.p.stdout.read().strip().decode("utf-8")
                    self.err = "" if ErrToOut else self.p.stderr.read().strip().decode("utf-8")
                except KeyboardInterrupt:
                    os.killpg(self.pid, 2)
                self.p.wait()
                self.status = self.p.returncode
        except OSError as e:
            Log("Got exception running {}: {}".format(Command, e))
            exit(1)

    def Wait(self):
        try:
            self.p.wait()
        except KeyboardInterrupt:
            os.killpg(self.pid, 2)
            self.p.wait()
            self.status = self.p.returncode
        except AttributeError:
            pass


def Telnet(Host, Port, Timeout=5):
    try:
        Sock = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)
        Sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        Sock.setsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY, 1)
        Sock.settimeout(Timeout)
        Sock.connect((Host, Port, 0, 0))
        Data = Sock.recv(4096).decode("utf-8")
        Sock.close()
        return Data
    except socket.error as e:
        Log("Failed to get data from {}:{:d} {}".format(Host, Port, e))
        return ""


# =============================================================================================================


def TestIfMyBackupDay():
    GroupHostsFile = "/etc/solomon/group_hosts"
    GroupHosts = [H.strip() for H in GetFileData(GroupHostsFile).split("\n") if len(H) > 0]
    if len(GroupHosts) == 0:
        Log("Bad group hosts file: no entries")
        exit(1)
    # Week number of the year
    BackupHost = GroupHosts[int(time.strftime("%W")) % len(GroupHosts)]
    Hostname = socket.gethostname()
    if BackupHost == Hostname:
        Log("It is my ({}) time to backup mysql db.".format(Hostname))
    else:
        Log("It is time for {} to backup mysql db, not {}.".format(BackupHost, Hostname))
        exit(0)


def MysqlBackup(Host, Port, FileName):
    global MysqlPassword

    if len(MysqlPassword) == 0:
        for L in GetFileData(SecretFile).split("\n"):
            LineList = [F.strip() for F in L.split(":")]
            if LineList[0] == "grafana":
                MysqlPassword = LineList[1]
                break
        if len(MysqlPassword) == 0:
            Log("Failed to get MySQL password from {}".format(SecretFile))
            exit(1)

    try:
        File = open(FileName, mode="wb")
    except OSError as e:
        Log("Failed to open {}: {}".format(FileName, e))
        exit(1)
    Cmd = [
        "/usr/bin/mysqldump",
        "--host={}".format(Host),
        "--port={}".format(Port),
        "--ssl-ca=".format(SSLCAFile),
        "--ssl-mode=REQUIRED",
        "--user=grafana",
        "--password={}".format(MysqlPassword),
        "grafana",
    ]
    M = Exec(Cmd, Pipe=True)
    G = Exec(["/bin/gzip"], Stdin=True, Stdout=File, Pipe=True)
    SkipLines = False
    Size = 0
    TimeStep = 10
    Timer = time.time() + TimeStep
    for Line in M.stdout:
        if b"GLOBAL.GTID_PURGED" in Line or b"SESSION.SQL_LOG_BIN" in Line:
            if b";" not in Line:
                SkipLines = True
            continue
        if SkipLines:
            if b";" in Line:
                SkipLines = False
            continue
        G.stdin.write(Line)
        Size += len(Line)
        Time = time.time()
        if Time > Timer:
            Timer = Time + TimeStep
            Log("{:d}Mb written (raw)".format(Size >> 20))
    M.Wait()
    G.stdin.close()
    G.Wait()
    File.close()


# =============================================================================================================


def main():
    Parser = argparse.ArgumentParser()
    Parser.add_argument("--dir", type=str, default="", metavar="PATH", help="directory for backups (default=%(default)s)")
    Args = Parser.parse_args()

    TestIfMyBackupDay()

    S = Telnet("::1", 3304)
    Log("Got mysql state: {}".format(S))
    MysqlEndpoints = {}
    ReEndpoint = re.compile(r"\[(?P<Host>[a-zA-Z0-9.-]+):(?P<Port>[0-9]+)\]:(?P<State>[A-Z_]+)")
    for E in ReEndpoint.finditer(S):
        State = E.group("State")
        MysqlEndpoints[State] = MysqlEndpoints.get(State, set()) | {(E.group("Host"), E.group("Port"))}
    if "RO" in MysqlEndpoints:
        BackupEndpoint = MysqlEndpoints["RO"].pop()
    elif "RW" in MysqlEndpoints:
        BackupEndpoint = MysqlEndpoints["RW"].pop()
    else:
        Log("No RO or RW mysql endpoints found.")
        exit(1)

    FileName = os.path.join(Args.dir, time.strftime("mysql_grafana_backup_%Y%m%d_%H%M%S.gz"))
    Log("Start backup from {}:{} to {}".format(*BackupEndpoint, FileName))
    MysqlBackup(*BackupEndpoint, FileName)
    Log("Done")


# =============================================================================================================


if __name__ == "__main__":
    main()
