#!/bin/sh

PLUGINS="
    alexanderzobnin-zabbix-app
    agenty-flowcharting-panel
    briangann-datatable-panel
    camptocamp-prometheus-alertmanager-datasource
    farski-blendstat-panel
    goshposh-metaqueries-datasource
    grafana-clock-panel
    grafana-piechart-panel
    grafana-polystat-panel
    grafana-worldmap-panel
    marcuscalidus-svg-panel
    michaeldmoore-multistat-panel
    natel-plotly-panel
    grafana-image-renderer
    simpod-json-datasource=0.2.6	# GRAFANA-268
    vertamedia-clickhouse-datasource
    vonage-status-panel
    yesoreyeram-boomtable-panel
"
NOSTARTFLAG="/Berkanavt/grafana/run/DONOTSTART"


log() {
	/bin/date "+%Y-%m-%d [%H:%M:%S] -  $*"
}

ask() {
	local _answer

	echo -n "$1   Break? [Y/n]: "
	read _answer
	if [ "$_answer" != "n" -a "$_answer" != "N" ] ; then
		echo "Exiting..."
		exit 0
	fi
}

grafana_is_healthy() {
	local _c

	_c="$(curl -qs -w "%{http_code}" -o /dev/null 'http://localhost:8080/api/health')"
	if [ "$_c" != "200" ] ; then
		return 1
	fi
	return 0
}

wait_grafana() {
	local _time=$1 _step=5

	while true ; do
		if [ $_time -lt 0 ] ; then
			log "FAILED to wait for grafana server to go UP. Exiting ..."
			exit 1
		fi
		if grafana_is_healthy ; then
			log "Grafana server is UP!"
			break
		else
			log "Grafana server is still DOWN. Waiting $_step seconds more ..."
		fi
		_time=$(($_time - $_step))
		sleep $_step
	done
}

# set default values, get command line
STOP=1
FORCE=0
GRAFANA_IS_ACTIVE=0
for i in "$@" ; do
	case "$i" in
		"-nostop")
			STOP=0
		;;
		"-force")
			FORCE=1
		;;
	esac
done

if ! which grafana-cli >/dev/null ; then
	echo
	echo "INSTALL PLUGINS WITH: "
	echo "grafana-cli plugins install <plugin>"
	echo
	exit
fi

# stop grafana-server if needed
if [ $STOP -eq 1 ] ; then
	if [ $FORCE -ne 1 ] ; then
		ask "Ready to stop grafana-server, install or update plugins."
	fi
	if systemctl is-active grafana-server > /dev/null; then
		log "Stopping grafana-server"
		GRAFANA_IS_ACTIVE=1
		touch $NOSTARTFLAG
		systemctl stop grafana-server
	else
		log "Not stopping grafana-server: not active"
	fi
fi

# install or upgrade plugins
PLUGINS="$(echo "$PLUGINS" | sed -E 's/[ \t]*#.*$//')"
INSTALLED_PLUGINS="$(grafana-cli plugins ls | awk '$2 == "@" {print $1 "=" $3}')"
for PV in $PLUGINS ; do
	P="$(echo $PV | sed -E 's/^([^=]+)(=(.+))?$/\1/')"
	V="$(echo $PV | sed -E 's/^([^=]+)(=(.+))?$/\3/')"
	V_LOCAL="$(echo "$INSTALLED_PLUGINS" | sed -nE '/^'$P'/s/^.+=(.+)$/\1/p')"
	V_LAST="$(grafana-cli plugins list-versions $P | head -1)"

	if [ -z "$V_LOCAL" ] ; then
		log "Installing plugin $P"
	elif [ "$V_LOCAL" = "$V" ] ; then
		log "Plugin $P has good version $V"
		continue
	elif [ "$V_LOCAL" = "$V_LAST" ] ; then
		log "Plugin $P has last version $V"
		continue
	else
		log "Update plugin $P"
	fi
	grafana-cli plugins install $P $V
done

# start grafana-server and wait for it if needed
if [ $STOP -eq 1 ] ; then
	if [ $GRAFANA_IS_ACTIVE -eq 1 ] ; then
		log "Starting grafana-server"
		rm -vf $NOSTARTFLAG
		systemctl start grafana-server
		wait_grafana 180
	else
		log "Not starting grafana-server: was not active"
	fi
fi
