#!/usr/bin/env python

import gevent
import os
import re
import json
import time

import system

sleep = gevent.sleep


# ==============================================================================================================


class TService(object):
    SYSV    = 0
    UPSTART = 1
    SYSTEMD = 2

    def __init__(self, Service, Process=None, ProcessFilter=None):
        self.Service       = Service
        self.StartSystem   = self.SetStartSystem()
        self.Process       = Service if self.StartSystem == self.SYSV and Process is None else Process
        self.Pid           = None
        self.Status        = None
        self.SubStatus     = None
        self.ProcessFilter = ProcessFilter
        self.LastStatusUpdateTime = 0
        self.StatusUpdate()

    def __dict__(self):
        return {
            "Service":       self.Service,
            "StartSystem":   self.GetStartSystem(),
            "Process":       self.Process,
            "Pid":           self.Pid,
            "Status":        self.Status,
            "SubStatus":     self.SubStatus,
            "ProcessFilter": self.ProcessFilter
        }

    def __str__(self):
        return json.dumps(self.__dict__(), sort_keys=True)

    def GetStartSystem(self):
        return "SysV"    if self.StartSystem == self.SYSV    else \
               "Upstart" if self.StartSystem == self.UPSTART else \
               "Systemd" if self.StartSystem == self.SYSTEMD else \
                self.StartSystem

    def StartSystemIsOk(self):
        self.StatusUpdate()
        if self.StartSystem == self.UPSTART:
            if ((self.Status == "start" and self.SubStatus in ["running", "killed"]) \
                or (self.Status == "stop" and self.SubStatus == "killed")) \
                and self.Pid is not None and self.Process is not None and system.GetCommandFromPid(self.Pid) != self.Process:
                return False
        return True

    def SetStartSystem(self):
        if (    os.path.exists("/etc/systemd/system/" + self.Service + ".service") \
            or  os.path.exists("/lib/systemd/system/" + self.Service + ".service")) \
            and os.path.exists("/bin/systemctl"):
            return self.SYSTEMD
        elif    os.path.exists("/etc/init/" + self.Service + ".conf") \
            and os.path.exists("/sbin/initctl"):
            return self.UPSTART
        elif os.path.exists("/etc/init.d/" + self.Service):
            return self.SYSV
        else:
            return None

    reInitdStatus = re.compile("(?<!not) running")
    def SetInitdPidStatus(self):
        Status = system.Exec(["/etc/init.d/" + self.Service, "status"]).out
        if "Active:" in Status and "Loaded:" in Status:
            self.SetSystemdPidStatus()
        else:
            self.Pid = system.PID(system.PGrep(self.Process, self.ProcessFilter))
            self.Status = "running" if self.reInitdStatus.search(Status) else "stopped"

    def SetUpstartPidStatus(self):
        Status = system.Exec(["/usr/sbin/service", self.Service, "status"]).out
        if "Active:" in Status and "Loaded:" in Status:
            self.SetSystemdPidStatus()
        else:
            self.Pid = system.PID(Status.split()[-1].strip())
            self.Status, self.SubStatus = Status.split()[1].rstrip(',').split('/')

    def SetSystemdPidStatus(self):
        Status = system.Exec(["/bin/systemctl", "show", "--no-pager", "--property=ActiveState,SubState,MainPID", self.Service]).out
        StatusDict = {}
        for S in Status.split('\n'):
            SList = S.split('=')
            StatusDict[SList[0]] = SList[1].strip()
        self.Pid = system.PID(StatusDict["MainPID"])
        self.Status = StatusDict["ActiveState"]
        self.SubStatus = StatusDict["SubState"]

    def StatusUpdate(self):
        Time = time.time()
        if Time - self.LastStatusUpdateTime < 0.1:
            return
        self.LastStatusUpdateTime = Time
        if self.StartSystem == self.SYSV:
            self.SetInitdPidStatus()
        elif self.StartSystem == self.UPSTART:
            self.SetUpstartPidStatus()
        elif self.StartSystem == self.SYSTEMD:
            self.SetSystemdPidStatus()
        else:
            self.Pid       = None
            self.Status    = None
            self.SubStatus = None

    #================================================================================================================
    # SysV:    status(Status): stopped running
    # Upstart:
    #          goal(Status): start stop
    #          status(SubStatus): waiting starting pre-start spawned post-start running pre-stop stopping killed post-stop
    # Systemd:
    #          status(Status): active inactive activating deactivating failed

    def isRunning(self):
        self.StatusUpdate()
        if self.StartSystem == self.SYSV and self.Status == "running":
            return (self.Process is not None and self.Pid is not None and system.GetCommandFromPid(self.Pid) == self.Process)
        elif self.StartSystem == self.UPSTART and self.Status == "start" and self.SubStatus == "running":
            return (self.Process is None or (self.Pid is not None and system.GetCommandFromPid(self.Pid) == self.Process))
        elif self.StartSystem == self.SYSTEMD and self.Status == "active":
            return (self.Process is None or (self.Pid is not None and system.GetCommandFromPid(self.Pid) == self.Process))
        else:
            return False

    def isStopped(self):
        self.StatusUpdate()
        if self.StartSystem == self.SYSV and self.Status == "stopped":
            return True
        elif self.StartSystem == self.UPSTART and self.Status == "stop" and self.SubStatus == "waiting":
            return True
        elif self.StartSystem == self.SYSTEMD and self.Status in ["inactive", "failed"]:
            return True
        else:
            return False

    def Stop(self, NotFullyStartedIsOk=False):
        def vStop(Name, Pid, Cmd):
            Process = system.GetCommandFromPid(Pid)
            system.Log("Trying to stop %s (process %s)" % (Name, Process))
            os.system(Cmd)
            if Process is not None:
                for c in range(0, 20):
                    sleep(0.5)
                    if system.GetCommandFromPid(Pid) != Process:
                        break
                    system.Log("Waiting for %.1f seconds (%s)..." % (c/2.0, Name))
                if c == 19:
                    system.Log("Service %s is too slow to stop without help, kill it!" % Name)
                    os.system("/bin/kill -9 " + str(Pid))
                    sleep(0.5)
        self.StatusUpdate()
        try:
            if self.StartSystem == self.SYSV and self.Status == "running":
                vStop(self.Service, self.Pid, "/etc/init.d/%s stop >/dev/null 2>&1 &" % self.Service)
            elif self.StartSystem == self.UPSTART and self.Status == "start" and (NotFullyStartedIsOk or self.SubStatus == "running"):
                vStop(self.Service, self.Pid, "/usr/sbin/service %s stop >/dev/null 2>&1 &" % self.Service)
            elif self.StartSystem == self.SYSTEMD and self.Status in ["active", "activating"] and (NotFullyStartedIsOk or self.Status != "activating"):
                vStop(self.Service, self.Pid, "/bin/systemctl stop %s >/dev/null 2>&1 &" % self.Service)
            else:
                system.Log("Need to stop %s, but current state is %s/%s!" % (self.Service, self.Status, self.SubStatus))
                return False
            return True
        except Exception as e:
            system.Log("Exception while stopping %s: %s" % (self.Service, e.message))

    def Start(self, NotFullyStoppedIsOk=False):
        def vStart(Name, Cmd):
            system.Log("Trying to start %s" % Name)
            os.system(Cmd)
            sleep(0.5)
        self.StatusUpdate()
        try:
            if self.StartSystem == self.SYSV and self.Status == "stopped":
                vStart(self.Service, "/etc/init.d/%s start >/dev/null 2>&1 &" % self.Service)
            elif self.StartSystem == self.UPSTART and self.Status == "stop" and (NotFullyStoppedIsOk or self.SubStatus == "waiting"):
                vStart(self.Service, "/usr/sbin/service %s start >/dev/null 2>&1 &" % self.Service)
            elif self.StartSystem == self.SYSTEMD and self.Status in ["inactive", "deactivating", "failed"] and (NotFullyStoppedIsOk or self.Status != "deactivating"):
                vStart(self.Service, "/bin/systemctl start %s >/dev/null 2>&1 &" % self.Service)
            else:
                system.Log("Need to start %s, but current state is %s/%s!" % (self.Service, self.Status, self.SubStatus))
                return False
            return True
        except Exception as e:
            system.Log("Exception while starting %s: %s" % (self.Service, e.message))
