import time

import system
import service


# ==============================================================================================================


class TServiceChecker(object):
    RESETWAIT    = 1800
    READYWAIT    = 300
    REBOOTWAIT   = 1200
    REBOOTCHECKS = 10
    UP           = 1
    DOWN         = 0

    def __init__(self, Service, Process=None, ProcessFilter=None, TargetState=UP, ReadyFuncList=[], PingFunction=None, ReadyWaitTime=READYWAIT, ResetWaitTime=RESETWAIT):
        Now = time.time()
        self.Service         = service.TService(Service, Process, ProcessFilter)
        self.TargetState     = TargetState
        self.ReadyFuncList   = ReadyFuncList
        self.PingFunction    = PingFunction
        self.ReadyWaitTime   = ReadyWaitTime
        self.ResetWaitTime   = ResetWaitTime
        self.LastRunningTime = Now
        self.LastReadyTime   = Now
        self.LastResetTime   = Now - system.NormalRand(self.ResetWaitTime*0.8)
        self.Ready           = None
        self.Running         = None
        self.ResetCounter    = 0
        self.LastKnownPid    = None
        self.LastOkStartSystemTime        = Now
        self.ChecksSinceLastStartSystemOk = 0
        self.StartSystemIsBad             = False
        self.StartSystemIsBadReboot       = True

    def Log(self, String):
        system.Log("[%s] %s" % (self.Service.Service, String))

    def SetTargetStateUp(self):
        self.Log("target state is set to UP")
        self.TargetState = self.UP

    def SetTargetStateDown(self):
        self.Log("target state is set to DOWN")
        self.TargetState = self.DOWN

    def TargetStateIsUp(self):
        return (self.TargetState == self.UP)

    def ResetCount(self):
        if self.LastKnownPid != self.Service.Pid:
            if not self.LastKnownPid is None:
                self.ResetCounter += 1
                self.LastResetTime = time.time()
            self.LastKnownPid = self.Service.Pid

    def StartSystemCheck(self):
        Now = time.time()
        if self.Service.StartSystemIsOk():
            self.LastOkStartSystemTime = Now
            self.ChecksSinceLastStartSystemOk = 0
        else:
            self.ChecksSinceLastStartSystemOk += 1
            self.Log("service is '%s/%s', and start system is turned bad (time=%.1f counter=%d)" % (
                self.Service.Status,
                self.Service.SubStatus,
                Now - self.LastOkStartSystemTime,
                self.ChecksSinceLastStartSystemOk
            ))
            if Now - self.LastOkStartSystemTime > self.REBOOTWAIT and self.ChecksSinceLastStartSystemOk > self.REBOOTCHECKS:
                self.StartSystemIsBad = True
                if self.StartSystemIsBadReboot:
                    system.KillServerReboot()

    def Check(self):
        self.Running = self.Service.isRunning()
        self.StartSystemCheck()
        if self.Running:
            ReadyFuncBootList = [{"name": ReadyFunc.__name__, "bool": ReadyFunc()} for ReadyFunc in self.ReadyFuncList]
            self.Ready = (sum([1 if D["bool"] else 0 for D in ReadyFuncBootList]) == len(self.ReadyFuncList))
            if len(ReadyFuncBootList) > 0:
                self.Log(", ".join(["%s: %s" % (D["name"], D["bool"]) for D in ReadyFuncBootList]))
        else:
            self.Ready = None

        Now = time.time()
        if self.Ready:
            self.LastReadyTime = Now
        if self.Running:
            self.LastRunningTime = Now

        if self.TargetState == self.UP:
            self.ResetCount()
            if not self.Running:
                if not self.Service.Start():
                    self.TryReset(ResetWaitTime=60, ResetReason="the service is not running and failed to start it")
            elif not self.Ready:
                if Now - self.LastReadyTime > self.ReadyWaitTime:
                    self.TryReset(ResetReason="the service is not ready for too long")
                elif not self.PingFunction is None:
                    self.PingFunction()
        elif self.Running and self.TargetState == self.DOWN:
            self.Service.Stop()

        self.Log("status: %s, running: %s, ready: %s, reset counter: %d, last reset: %.3fs(%.1fs), last ready: %.3fs(%.1fs)" % (
            self.Service,
            self.Running,
            self.Ready,
            self.ResetCounter,
            Now - self.LastResetTime, self.ResetWaitTime,
            Now - self.LastReadyTime, self.ReadyWaitTime
        ))

    def TryReset(self, PreStartFunc=None, ResetWaitTime=None, ResetReason="I just want to"):
        dt = time.time() - self.LastResetTime
        if ResetWaitTime is None:
            ResetWaitTime = self.ResetWaitTime
        Text = "ResetWaitTime = %d sec, running time = %d sec" % (ResetWaitTime, dt)
        if dt > ResetWaitTime:
            system.SysLog(self.Service.Service, "WATCHDOG: resetting %s because %s. %s" % (self.Service.Service, ResetReason, Text))
            self.Log("resetting because %s. %s" % (ResetReason, Text))
            self.Service.Stop()
            if PreStartFunc != None:
                PreStartFunc()
            self.Service.Start()
            self.LastResetTime = time.time()
            return True
        else:
            self.Log("skipped reset: %s" % Text)
        return False
