#!/usr/bin/env python

import json
import socket
import re
import sys
import ast
import operator
import argparse
import urllib2


HOSTCONFIG = "/Berkanavt/portos/config/host.conf"
PORTOSCONF = "/Berkanavt/portos/config/portos.conf"


def GetHostname():
    return socket.gethostname()


def Conductor(Macros):
    URL = "http://c.yandex-team.ru/api/groups2hosts/%s" % Macros
    try:
        return urllib2.urlopen(URL, timeout=5).read().strip().split()
    except Exception as e:
        print("Failed to resolve conductor macro '%s'. %s" % (Macros, e))
        exit(1)


def ReadFile(FileName):
    try:
        File = open(FileName)
        Data = File.read().decode("utf-8")
        File.close()
        if len(Data) > 0:
            return Data
    except IOError as e:
        if e.errno == 2 or e.errno == 3:
            pass
        else:
            raise
    return ""


def WriteFile(Name, Content, Mode='w'):
    with open(Name, Mode) as File:
        File.write(Content)


reInt = re.compile("(-?[0-9]+)")
def Int(Str, Default=0):
    S = reInt.search(Str)
    return int(S.group(1)) if S else Default


ReBracedDots = re.compile("^(-?[0-9]+)\.\.(-?[0-9]+)(?:\.\.-?([0-9]+))?$")
def UnwindString(String, NoTokens=True):
    def OpenRange(Begin, End, Step=None):
        Pad   = any([S.startswith(("0", "-0")) for S in [Begin, End] if S not in ["0", "-0"]])
        Fmt   = ("%%0%dd" % max(len(Begin), len(End))) if Pad else "%d"
        Begin = int(Begin)
        End   = int(End)
        Step  = (int(Step) or 1) if Step else 1
        Range = xrange(Begin, End + 1, Step) if Begin < End else xrange(Begin, End - 1, -Step)
        return [Fmt % N for N in Range]

    def ParseExpression(String):
        M = ReBracedDots.match(String)
        if M:
            return OpenRange(*M.groups())
        L, Lp = [], [""]
        Begin, Depth = 0, 0
        for Idx, C in enumerate(String):
            if C in "{},":
                if C == "{":
                    if Depth == 0:
                        if Begin < Idx:
                            Lp = [P + String[Begin:Idx] for P in Lp]
                        Begin = Idx + 1
                    Depth += 1
                elif C == "}":
                    Depth -= 1
                    if Depth == 0:
                        Lp = [P + E for P in Lp for E in ParseExpression(String[Begin:Idx])]
                        Begin = Idx + 1
                elif Depth == 0:
                    L += [P + String[Begin:Idx] for P in Lp] if Begin < Idx else Lp
                    Lp = [""]
                    Begin = Idx + 1
        L += [P + String[Begin:] for P in Lp] if Begin < len(String) else Lp
        if len(L) == 1:
            return ["{" + L[0] + "}"]
        return L

    def GetTokens(String):
        L, Stack = [], []
        for Idx, C in enumerate(String):
            if C in "{}":
                if C == '{':
                    Stack.append(Idx)
                else:
                    if len(Stack) > 0:
                        B = Stack.pop()
                        if Idx - B > 1:
                            L = [T for T in L if T[0] < B]
                            L.append([B, Idx])
        if len(L) == 0:
            return [String]
        Pre = 0
        Lx = []
        for Begin, End in L:
            if Pre < Begin:
                Lx.append(String[Pre:Begin])
            Lx.append(ParseExpression(String[Begin + 1:End]))
            Pre = End + 1
        if Pre < len(String):
            Lx.append(String[Pre:])
        return Lx

    def ListsProduct(Lists):
        R = [[]]
        for L in Lists:
            R = [X + [Y] for X in R for Y in L]
        return R

    def MultiplyTokens(Tokens, NoTokens=True):
        C = 0
        TokensDraft = []
        for T in Tokens:
            if isinstance(T, list):
                TokensDraft.append(C)
                C += 1
            else:
                TokensDraft.append(T)
        L = []
        for P in ListsProduct([T for T in Tokens if isinstance(T, list)]):
            S = "".join([P[T] if isinstance(T, int) else T for T in TokensDraft])
            L.append(S if NoTokens else (S, P))
        return L

    return MultiplyTokens(GetTokens(String), NoTokens)


ReToken = re.compile("\([^\(\)]+\)")
ReFmt = re.compile("{[^{}:]:([^{}]+)}")
def Math(Str, *Args, **Kwargs):
    Operators = {
        ast.Add:    operator.add,
        ast.Sub:    operator.sub,
        ast.Mult:   operator.mul,
        ast.Div:    operator.truediv,
        ast.BitXor: operator.xor,
        ast.USub:   operator.neg
    }
    def Eval(Expr):
        def Eval_(Node):
            if isinstance(Node, ast.Num):
                return Node.n
            elif isinstance(Node, ast.BinOp):
                return Operators[type(Node.op)](Eval_(Node.left), Eval_(Node.right))
            elif isinstance(Node, ast.UnaryOp):
                return Operators[type(Node.op)](Eval_(Node.operand))
            else:
                raise TypeError(Node)
        return Eval_(ast.parse(Expr, mode="eval").body)

    def GetFmt(Str):
        Fmt = ""
        while True:
            M = ReFmt.search(Str)
            if not M:
                return Str, Fmt
            if Fmt == "":
                Fmt = Str[M.start(1):M.end(1)]
            Str = Str[:M.start(1)] + Str[M.end(1):]

    while True:
        M = ReToken.search(Str)
        if not M:
            return Str.format(*Args, **Kwargs)
        CalcFmt, Fmt = GetFmt(Str[M.start(0) + 1:M.end(0) - 1])
        CalcStr = CalcFmt.format(*map(Int, Args), **Kwargs)
        EvalStr = "{:{Fmt}}".format(Eval(CalcStr), Fmt=Fmt)
        Str = Str[:M.start(0)] + EvalStr + Str[M.end(0):]


def UnwrapMap(Map):
    M = {}
    for HostZ, ContZDict in Map.items():
        ContList = [(Cont, Tmpl) for ContZ, Tmpl in ContZDict.items() for Cont in UnwindString(ContZ, True)]
        ContC = 0
        HostList = []
        for Host, Tokens in UnwindString(HostZ, False):
            if Host.startswith("CG@"):
                HostList += [(H, Tokens) for H in sorted(Conductor(Host[3:]))]
            else:
                HostList.append((Host, Tokens))
        for HostC, (Host, Tokens) in enumerate(HostList):
            ContDict = {}
            for ContHC, (Cont, Tmpl) in enumerate(ContList):
                # special kwargs:
                #       H  = host counter
                #       HC = container counter for each host
                #       C  = container counter for host group
                ContDict[Math(Cont, *Tokens, H=HostC, HC=ContHC, C=ContC)] = Tmpl
                ContC += 1
            M[Host] = ContDict
    return M


def main():
    STDOUT = "/dev/stdout"
    Hostname = GetHostname()
    if len(Hostname) == 0:
        print("Cannot get my hostname")
        return
    Parser = argparse.ArgumentParser(description="create portos.conf from hosts.conf")
    Parser.add_argument("-i", "--input", type=str, required=False, default=[HOSTCONFIG], metavar="FILE", help="input config to use (default=%(default)s)", nargs=1)
    Parser.add_argument("-o", "--output", type=str, required=False, default=[PORTOSCONF], metavar="FILE", help="output config to use (default=%(default)s)", nargs=1)
    Parser.add_argument("-n", "--name", type=str, required=False, default=[Hostname], metavar="HOSTNAME", help="hostname to get config for (default=%(default)s)", nargs=1)
    Parser.add_argument("-a", "--all", action="store_true", help="dump all config file to stdout")
    Args = Parser.parse_args()
    Input    = Args.input[0]
    Output   = Args.output[0]
    Hostname = Args.name[0]
    All      = Args.all
    if Output == "-":
        Output = STDOUT

    Conf = ReadFile(Input)
    if len(Conf) == 0:
        print("Cannot load from %s" % Input)
        return
    ConfDict = json.loads(Conf)
    ResultDict = {}
    ConfDict["map"] = UnwrapMap(ConfDict.get("map", {}))
    if All:
        print json.dumps(ConfDict, sort_keys=True, indent=4)
        return
    for Key, Value in ConfDict["map"].get(Hostname, {}).items():
        if isinstance(Value, dict):
            ResultDict[Key] = Value
        else:
            if Value not in ConfDict["templates"]:
                print("Cannot find %s template in %s" % (Value, Input))
                return
            ResultDict[Key] = ConfDict["templates"][Value]
    if len(ResultDict) == 0:
        print("Got empty config for %s" % (Hostname))
        return
    if Output != STDOUT:
        print("Writing to file %s" % Output)
    WriteFile(Output, json.dumps(ResultDict, sort_keys=True, indent=4))


if __name__ == "__main__":
    main()
