#!/usr/bin/env python3

import sys
import os
import struct
import socket
import fcntl
import datetime
import subprocess
import hashlib
import json


# ==================================================================================================================


def Log(String, File=sys.stdout):
    print("{} - {}".format(datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S.%f"), String), file=File)


def LogErr(String):
    Log(String, File=sys.stderr)


class Exec:
    def __init__(self, Command, Env={"LANG": "C"}, ErrToOut=False, Stdin=None):
        self.pid = -1
        self.out = ""
        self.err = ""
        self.status = 1
        try:
            self.p = subprocess.Popen(
                Command,
                env=Env,
                stdin=subprocess.PIPE if Stdin is True else Stdin,
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT if ErrToOut else subprocess.PIPE,
                close_fds=True,
                preexec_fn=lambda: os.setpgid(os.getpid(), os.getpid())
            )
            self.pid = self.p.pid
            try:
                self.out = self.p.stdout.read().strip().decode("utf-8")
                self.err = "" if ErrToOut else self.p.stderr.read().strip().decode("utf-8")
            except KeyboardInterrupt:
                os.killpg(self.pid, 2)
            self.p.wait()
            self.status = self.p.returncode
        except OSError as e:
            LogErr("Got exception running {}: {}".format(Command, e))
            exit(1)


# ==============================================================================================================


def GetFileData(FileName):
    Data = ""
    try:
        File = open(FileName, 'r')
        Data = File.read()
        File.close()
    except (IOError, OSError) as e:
        if e.errno == 2 or e.errno == 3:
            pass
        else:
            raise
    return Data


def PutFileData(FileName, Data, Mode=0o644, Atomic=False):
    TmpFile = "{}.tmp".format(FileName)
    try:
        File = os.open(TmpFile if Atomic else FileName, os.O_WRONLY | os.O_CREAT, Mode)
        os.write(File, Data.encode("utf-8"))
        os.close(File)
        if Atomic:
            os.replace(TmpFile, FileName)
    except OSError as e:
        if e.errno == 3:
            pass
        else:
            raise


class LockFile:
    def __init__(self, FileName):
        self.FileName = FileName
        self.File = None

    def Lock(self, Block=False):
        try:
            self.File = open(self.FileName, 'a')
            fcntl.lockf(self.File, fcntl.LOCK_EX | (0 if Block else fcntl.LOCK_NB))
        except Exception as e:
            LogErr("Cannot lock {}: {}".format(self.FileName, e))
            return False
        return True

    def UnLock(self):
        try:
            self.File.close()
        except Exception:
            return False
        return True


# ==============================================================================================================


def GetHostname():
    return socket.gethostname()


def IPv6ToNum(Addr):
    Hi, Lo = struct.unpack("!QQ", socket.inet_pton(socket.AF_INET6, Addr))
    return (Hi << 64) | Lo


def NumToIPv6(N):
    return socket.inet_ntop(socket.AF_INET6, struct.pack("!QQ", N >> 64, N & 0xffffffffffffffff))


def ResolveNameToIPv6(FQDN):
    try:
        return [i[-1] for i in socket.getaddrinfo(FQDN, 0) if i[0] == socket.AF_INET6][0][0]
    except (IndexError, socket.gaierror) as e:
        LogErr("Failed to resolve '{}': {}".format(FQDN, e))


def GetProjectIdFromIPv6(IPv6):
    IPv6Num = IPv6ToNum(IPv6)
    return (IPv6Num >> 32) & 0xffffffff


def SetIdToIPv6(IPv6, Id, Length=32, Offset=64):
    if Length + Offset > 128 or Length <= 0 or Offset < 0:
        LogErr("Bad id length={} offset={} for ip={}".format(Length, Offset, IPv6))
        exit(1)
    IPv6Num = IPv6ToNum(IPv6)
    Mask = (1 << Length) - 1
    Shift = 128 - Offset - Length
    return NumToIPv6((IPv6Num & ~(Mask << Shift)) | ((Id & Mask) << Shift))


def StringToId(String, Length=16):
    Hi, Lo = struct.unpack("!QQ", hashlib.md5(String.encode("utf-8")).digest())
    N = (Hi << 64) | Lo
    Result = N
    while N > 0:
        N >>= Length
        Result ^= N
    return Result & ((1 << Length) - 1)


def GetGlobalIPv6ForIface(Iface):
    AddrList = []
    for AddrLine in Exec(("/sbin/ip -o -6 addr list dev {} scope global".format(Iface)).split()).out.split('\n'):
        AddrLineArray = AddrLine.split()
        if len(AddrLineArray) > 3:
            AddrList.append(AddrLineArray[3].split('/')[0])
    return AddrList


def SetIPv6Iface(IPv6WithPrefix, Iface):
    return Exec(("/sbin/ip addr add {} dev {}".format(IPv6WithPrefix, Iface)).split()).status == 0


# ==============================================================================================================


def GetKuberSubnet():
    IdLength = 20
    try:
        Hostname = GetHostname()
        HostIPv6 = ResolveNameToIPv6(Hostname)
        HostIPv688List = GetGlobalIPv6ForIface("vlan688")
        if len(HostIPv688List) > 0:
            HostProjectId = GetProjectIdFromIPv6(HostIPv6)
            HostIPv688 = SetIdToIPv6(HostIPv688List[0], HostProjectId, Length=32, Offset=64)
            MyIPv6Draft = HostIPv688
            if HostIPv688 not in HostIPv688List:
                if not SetIPv6Iface("{}/96".format(HostIPv688), "vlan688"):
                    LogErr("Failed to set my projectId IPv6={} to vlan688".format(HostIPv688))
                    exit(1)
        else:
            MyIPv6Draft = HostIPv6
        PrefixLength = 64 + 32 + IdLength
        HostId = StringToId(Hostname, Length=IdLength)
        IPv6WithHostId = SetIdToIPv6(MyIPv6Draft, HostId, Length=IdLength, Offset=96)
        Subnet = SetIdToIPv6(IPv6WithHostId, 0, Length=(32 - IdLength), Offset=PrefixLength)
        Gateway = NumToIPv6(IPv6ToNum(Subnet) + 1)
        SubnetWithPrefixLength = "{}/{}".format(Subnet, PrefixLength)
        return Gateway, SubnetWithPrefixLength
    except Exception as e:
        LogErr("Failed to generate MTN subnet for CNI: {}".format(e))
        exit(1)


def Main():
    FileName = sys.argv[1] if len(sys.argv) == 2 else None
    Lock = LockFile("/var/lock/cni_mtn.lock")
    if not Lock.Lock():
        LogErr("Failed to lock")
        exit(0)
    Gateway, Subnet = GetKuberSubnet()
    if Subnet is None:
        exit(1)
    LogErr("Got MTN subnet {}".format(Subnet))

    Conf = json.dumps({
        "cniVersion": "0.4.0",
        "name": "knet",
        "type": "bridge",
        "bridge": "cni_bridge0",
        "isDefaultGateway": True,
        "hairpinMode": True,
        "mtu": 9000,
        "ipam": {
            "type": "host-local",
            "ranges": [
                [
                    {
                        "subnet": Subnet,
                        "gateway": Gateway
                    }
                ]
            ]
        }
    }, sort_keys=True, indent=4)

    if FileName:
        PutFileData(FileName, Conf, Atomic=True)
    else:
        print(Conf)
    Lock.UnLock()


if __name__ == "__main__":
    Main()
