#!/bin/sh -e

MACROS="_YANDEXNETS_ _DYNAMICWIFI_"
OVERLAY_NETS="2a0d:d6c0:200:200::/56"
UNDERLAY_NETS="2a0d:d6c0:200:100::/57"
LOCALHOST="127.0.0.1 ::1"

CONF="/etc/nginx/only_yandex_nets.conf"
MAX_CHANGES_PERCENT=10


do_conf() {
	for m in $MACROS ; do
		echo "# $m"
		curl -s --retry 2 --retry-delay 5 -m 30 "https://hbf.yandex.net/macros/${m}?format=text" | \
		sort -n | \
		awk '
			/^([0-9]{1,3}\.){3}[0-9]{1,3}\/[0-9]{1,2}$/ || 
			/^([0-9a-f]{1,4}:){1,7}(:|[0-9a-f]{1,4}|(:[0-9a-f]{1,4}){1,7})\/[0-9]{1,3}$/ {
				print "allow " $1 ";"
		}'
		echo ""
	done

	echo "# Private cloud (OVERLAY, UNDERLAY)"
	echo "allow $OVERLAY_NETS;"
	echo "allow $UNDERLAY_NETS;"
	echo ""

	echo "# localhost"
	for h in $LOCALHOST ; do 
		echo "allow $h;"
	done
	echo ""
	echo "# drop rest of the world"
	echo "deny all;"
}

nginx_reload() {
	local _bin="/usr/sbin/nginx" _defaults="/etc/default/nginx" _conf _pid

	_pid=$(cat /var/run/nginx.pid 2>/dev/null)
	if [ ${#_pid} -eq 0 ] ; then
		_pid=$(pgrep -ox nginx)
		if [ ${#_pid} -eq 0 ] ; then
			echo "Nginx is not running, cannot reload it"
			return
		fi
	fi

	if [ -r $_defaults ]; then
		. $_defaults
	fi
	_conf=${CONFFILE:-"/etc/nginx/nginx.conf"}
	if ! $_bin -t -c $_conf 2>/dev/null ; then
		echo "Nginx configuration test failed..."
		$_bin -t -c $_conf
		return
	fi
	echo "Reloading nginx configuration"
	kill -s HUP $_pid
}


if [ "$1" = "force" ] ; then
	FORCE="yes"
fi
do_conf > ${CONF}.tmp

LINES=$(diff ${CONF}.tmp ${CONF} 2>/dev/null | awk '/^[<>]/ {++N} END {print +N}')
if [ $LINES -eq 0 ] ; then
	if [ -f ${CONF} ] ; then
		echo "Nothing changed for ${CONF}"
		rm -f ${CONF}.tmp
	else
		echo "Destination file ${CONF} does not exist. Replacing it"
		mv -v ${CONF}.tmp ${CONF}
	fi
else
	CHANGES_PERCENT=$(awk -vL=$LINES 'END {print int(100*L/FNR)}' ${CONF})
	if [ $CHANGES_PERCENT -lt $MAX_CHANGES_PERCENT ] ; then
		echo "Few changes in ${CONF}: ${CHANGES_PERCENT}%, updating and reloading nginx!"
	elif [ -n "$FORCE" ]; then
		echo "Too many changes in ${CONF}: ${CHANGES_PERCENT}% > ${MAX_CHANGES_PERCENT}%, forcing nginx reload!"
	else
		echo "Too many changes in ${CONF}: ${CHANGES_PERCENT}% > ${MAX_CHANGES_PERCENT}%, not updating!"
		# rm -f ${CONF}.tmp
		exit
	fi
	mv -v ${CONF}.tmp ${CONF}
	set +e
	nginx_reload
fi
