#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os
import json
import re
import time
import argparse
import fcntl
import subprocess
import urllib.request
import urllib.parse
import urllib.error
import OpenSSL


Debug = False

GoodMethods = {"GET", "HEAD", "POST", "PUT", "DELETE", "CONNECT", "OPTIONS", "TRACE", "PATCH"}
GoodCodes = {
    "101", "102", "103",
    "200", "201", "202", "203", "204", "205", "206", "207", "208", "226",
    "300", "301", "302", "303", "304", "305", "306", "307", "308",
    "400", "401", "402", "403", "404", "405", "406", "407", "408", "409", "410", "411", "412", "413", "414", "415", "416", "417",
    "421", "422", "423", "424", "425", "426", "428", "429", "431", "444", "451", "494", "495", "496", "497", "499",
    "500", "501", "502", "503", "504", "505", "506", "507", "508", "510", "511"
}


class Exec:
    def __init__(self, Command, Env=None):
        self.pid = -1
        self.out = ""
        self.status = 1
        try:
            self.p = subprocess.Popen(
                Command,
                env=Env,
                stdout=subprocess.PIPE,
                close_fds=True,
                preexec_fn=lambda: os.setpgid(os.getpid(), os.getpid())
            )
            self.pid = self.p.pid
            try:
                self.out = self.p.stdout.read().strip().decode("utf-8")
            except KeyboardInterrupt:
                os.killpg(self.pid, 2)
            self.p.wait()
            self.status = self.p.returncode
        except OSError as e:
            print("ERROR: got exception running {}: {}".format(Command, e))
            self.status = 255


class GetURL:
    def __init__(self, URL, Headers={}, Data=None, Timeout=5, Method=None):
        self.code = 0
        self.err  = ""
        self.text = ""
        try:
            if isinstance(Data, dict):
                Data = urllib.parse.urlencode(Data)
            if isinstance(Data, str):
                Data = Data.encode("ascii")
            Req  = urllib.request.Request(URL, data=Data, headers=Headers, method=Method)
            Resp = urllib.request.urlopen(Req, timeout=Timeout)
            self.text = Resp.read().decode("utf-8")
            self.code = Resp.getcode()
        except urllib.error.HTTPError as e:
            self.code = e.code
            self.err  = e.reason
            print("HTTPError in GetURL for {} ({}) - {}".format(URL, self.code, self.err))
        except urllib.error.URLError as e:
            self.err = e.reason
            print("URLError in GetURL for {} - {}".format(URL, e))
        except Exception as e:
            self.err = str(e)
            print("General exception in GetURL for {} - {}".format(URL, e))


class TJsonDB:
    def __init__(self, FileName):
        self.FileName = FileName
        self.File = None
        self.InternalData = {"Data": {}, "Time": 0.0}
        self.Data = {}

    def __del__(self):
        self.Close()

    def TimeDelta(self):
        return time.time() - self.InternalData["Time"]

    def Open(self):
        try:
            self.File = open(self.FileName, "a+")
            fcntl.lockf(self.File.fileno(), fcntl.LOCK_EX | fcntl.LOCK_NB)
            self.File.seek(0, os.SEEK_SET)
            RawData = self.File.read()
            if len(RawData) > 0:
                try:
                    self.InternalData = json.loads(RawData)
                    self.Data = self.InternalData["Data"]
                except ValueError:
                    print("ERROR: corrupted DB in {}".format(self.FileName))
            self.File.seek(0, os.SEEK_SET)
            self.File.truncate()
            return True
        except IOError as e:
            print("ERROR: cannot access DB in {}: {}".format(self.FileName, e))
            return False

    def Close(self):
        try:
            self.InternalData["Time"] = time.time()
            json.dump(self.InternalData, self.File)
            self.File.close()
        except (AttributeError, ValueError):
            print("ERROR: cannot close DB file {}".format(self.FileName))


def NginxLog(DB, LogFile):
    Files = []
    HeadKey = "HEAD:" + LogFile
    SeekKey = "SEEK:" + LogFile
    Head = DB.Data.get(HeadKey)
    Seek = DB.Data.get(SeekKey)
    for FileName in [LogFile, LogFile + ".1"]:
        if os.path.isfile(FileName):
            File = open(FileName)
            FileHead = File.readline()
            Files.append((File, FileHead))
            if Head is None:
                File.seek(0, os.SEEK_END)
                break
            elif FileHead == Head:
                File.seek(Seek)
                break
            else:
                File.seek(0, os.SEEK_SET)
    while len(Files) > 0:
        File, DB.Data[HeadKey] = Files.pop()
        for Line in File:
            yield Line
        DB.Data[SeekKey] = File.tell()
        File.close()


def GetAccessCodesDict(DB, LogsFile, ReURLs):
    """
    :param DB: TJsonDB
    :param LogsFile: Path
    :return: CodesDict: { "URL": { "Method": { "HttpCode": <Count>, ... }, ... }, ... }
    """
    SchemaKey = "AccessMetricsSchema"
    Schema = DB.Data.get(SchemaKey, {})
    CodesDict = {U: {M: {C: 0.0 for C in LM} for M, LM in DU.items()} for U, DU in Schema.items()}
    ToupleDict = {}
    LineLists = (L.split(None, 9) for L in NginxLog(DB, LogsFile))
    LineTouples = ((L[6], L[5][1:], L[8]) for L in LineLists if len(L) > 8 and L[8].isdigit())
    for URL, Method, Code in LineTouples:
        if Method not in GoodMethods:
            Method = "BAD"
        if Code not in GoodCodes:
            Code = "000"
        for Re in ReURLs:
            M = Re.match(URL)
            if M:
                U = ":stub".join([
                    URL[M.start(0):M.start(1)]
                ] + [
                    URL[M.end(Idx):M.start(Idx + 1)] for Idx in range(1, M.lastindex)
                ] + [
                    URL[M.end(M.lastindex):M.end(0)]
                ]) if M.lastindex else URL[M.start(0):M.end(0)]
                if Debug:
                    print("URL regex match access line ({}, {}, {}): {}".format(Code, Method, U, URL))
                T = (U, Method, Code)
                break
        else:
            T = ("other", Method, Code)
        ToupleDict[T] = ToupleDict.get(T, 0.0) + 1.0
    for (U, Method, Code), Value in ToupleDict.items():
        for UX in [U, "total"]:
            DM = CodesDict.setdefault(UX, {Method: {Code: 0.0}}).setdefault(Method, {Code: 0.0})
            DM[Code] = DM.get(Code, 0.0) + Value
    DB.Data[SchemaKey] = {U: {M: [C for C in DM] for M, DM in DU.items()} for U, DU in CodesDict.items()}
    return CodesDict


reError = re.compile("failed \((\d+:\s[\w\s]*)\)")
def GetErrorLinesDict(DB, LogsFile):
    """
    :param DB: TJsonDB
    :param LogsFile: Path
    :return: CodesDict: { "CodeString": <Count>, ... }
    """
    SchemaKey = "ErrorMetricsSchema"
    Schema = DB.Data.get(SchemaKey, [])
    CodesDict = {C: 0 for C in Schema}
    for Line in NginxLog(DB, LogsFile):
        MatchObject = reError.search(Line)
        if MatchObject:
            Key = MatchObject.group(1)
            CodesDict[Key] = CodesDict.get(Key, 0.0) + 1.0
    DB.Data[SchemaKey] = list(CodesDict)
    return CodesDict


reCert = re.compile("^\s*ssl_certificate\s+([^ ;]+)\s*;\s*$")
def FindCerts(Basepath):
    Files = (os.path.join(D, F) for D, _, Fs in os.walk(Basepath) for F in Fs)
    CertMatches = (reCert.match(L) for F in Files if os.path.isfile(F) for L in open(F, mode="r", encoding="utf-8"))
    return set(M.group(1) for M in CertMatches if M)


def CertsTTL(Basepath):
    for CertFile in FindCerts(Basepath):
        if os.path.isfile(CertFile):
            Cert = OpenSSL.crypto.load_certificate(OpenSSL.crypto.FILETYPE_PEM, open(CertFile).read())
            ExpiredAt = time.mktime(time.strptime(Cert.get_notAfter().decode("utf-8"), "%Y%m%d%H%M%SZ"))
            TTLDays = (ExpiredAt - time.time())/86400
            yield CertFile, TTLDays


def GetNginxMetrics(DB, Coef):
    """
    :param DB: TJsonDB
    """
    DataKey = "NginxMetricsData"
    Data = DB.Data.get(DataKey, {})
    MetricsDict = {}
    R = Exec([
        "/bin/systemctl", "show",
        "--no-pager", "--no-legend",
        "--property=TasksCurrent,CPUUsageNSec,MemoryCurrent,MainPID", "nginx"
    ])
    if R.status != 0:
        return MetricsDict
    for S in R.out.split('\n'):
        SList = S.split('=')
        N = 0
        try:
            N = int(SList[1].strip())
        except Exception as e:
            print("ERROR: parsing systemctl response - {}".format(e))
        MetricsDict[SList[0]] = N
    Pid = MetricsDict.get("MainPID", 0)
    Tasks = MetricsDict.get("TasksCurrent", 1) if Pid > 0 else 0
    CPUUsageNSec = MetricsDict.get("CPUUsageNSec", 0)
    CPUUsageCores = (CPUUsageNSec - Data.get("CPUUsageNSec", CPUUsageNSec))/Coef/1000000000
    MetricsDict = {
        "memory_usage": MetricsDict.get("MemoryCurrent", 0) if Pid > 0 else 0,
        "workers": Tasks - 1 if Pid > 0 else 0,
        "cores_per_worker": (CPUUsageCores/((Tasks - 1) if Tasks > 1 else 1)) if Pid > 0 else 0
    }
    DB.Data[DataKey] = {"CPUUsageNSec": CPUUsageNSec}

    R = GetURL("http://localhost:28019/basic_status")
    if R.code == 200:
        Arr = R.text.split()
        try:
            MetricsDict.update({
                "connections.active": int(Arr[2]),
                "connections.accepted": int(Arr[7]),
                "connections.handled": int(Arr[8]),
                "requests_total": int(Arr[9]),
                "connections.reading": int(Arr[11]),
                "connections.writing": int(Arr[13]),
                "connections.waiting": int(Arr[15])
            })
        except Exception as e:
            print("ERROR: parsing nginx status response - {}".format(e))
    return MetricsDict


def ReadConfig(Path):
    try:
        if os.path.exists(Path):
            File = open(Path, 'r')
            try:
                return json.load(File)
            except ValueError:
                print("ERROR: corrupted DB '{}'".format(Path))
            finally:
                File.close()
    except IOError:
        print("ERROR: cannot open DB file '{}'".format(Path))
    return {}


def Main():
    NginxDir = "/etc/nginx"
    DBFile = "/run/shm/nginx_mon.json"
    JsonFile = "/usr/local/www/mon/nginx_mon.json"
    ConfigFile = "/usr/local/etc/nginx_mon.conf"
    AccessLog = "/var/log/nginx/access.log"
    ErrorLog = "/var/log/nginx/error.log"
    """
    Config:
    {
        "Time": <time interval between exec>,
        "UrlRegexes": ["<prefix1>", "<prefix2>", ... ]
    }
    """
    Parser = argparse.ArgumentParser()
    Parser.add_argument("-t", dest="time", default=[], nargs=1, type=float, help="interval between script starts to output RPS, seconds")
    Parser.add_argument("-u", "--urls-re", default=[], nargs="*", type=str, help="collect metrics for these URLs separately")
    Parser.add_argument("-c", "--config", default=[ConfigFile], nargs=1, type=str, help="config file path (default=%(default)s)")
    Parser.add_argument("-p", "--db-path", default=[DBFile], nargs=1, type=str, help="internal DB file path (default=%(default)s)")
    Parser.add_argument("-r", "--report-path", default=[JsonFile], nargs=1, type=str, help="report file path (default=%(default)s)")
    Parser.add_argument("-a", "--access-log", default=[AccessLog], nargs=1, type=str, help="access log path (default=%(default)s)")
    Parser.add_argument("-e", "--error-log", default=[ErrorLog], nargs=1, type=str, help="error log path (default=%(default)s)")
    Parser.add_argument("-d", "--debug", action="store_true", help="print debug info")
    Args = Parser.parse_args()

    globals()["Debug"] = Args.debug
    JsonFile = Args.report_path[0]
    AccessLog = Args.access_log[0]
    ErrorLog = Args.error_log[0]

    Config = ReadConfig(Args.config[0])
    if Debug:
        print("Config: {}".format(Config))
    ReURLs = [re.compile(Re) for Re in Args.urls_re]
    if len(ReURLs) == 0 and "UrlRegexes" in Config:
        for Re in Config["UrlRegexes"]:
            ReURLs.append(re.compile(Re))
    if Debug:
        print("URL regexes: {}".format(json.dumps([Re.pattern for Re in ReURLs])))

    DB = TJsonDB(Args.db_path[0])
    if not DB.Open():
        return

    AccessDict = GetAccessCodesDict(DB, AccessLog, ReURLs)
    ErrorDict = GetErrorLinesDict(DB, ErrorLog)
    Coef = DB.TimeDelta()

    if "Time" in Config:
        try:
            Coef = int(Config["Time"])
        except (ValueError, TypeError):
            pass
    if len(Args.time) == 1:
        Coef = Args.time[0]
    if Debug:
        print("Time coefficient is {:.2f}".format(Coef))

    metrics = [
        {
            "labels": {
                "sensor": "http.server.{}".format(L)
            },
            "value": V
        } for L, V in GetNginxMetrics(DB, Coef).items()
    ]
    metrics += [
        {
            "labels": {
                "sensor": "http.server.requests.status",
                "endpoint": U,
                "method": M,
                "code": C
            },
            "value": V/Coef
        } for U, DU in AccessDict.items()
        for M, DM in DU.items()
        for C, V in DM.items()
    ]
    if Debug:
        print("Access metrics: {}".format(json.dumps({
            U: {
                M: {
                    C: "{:.1f}/{:.2f} = {:.6f}".format(V, Coef, V/Coef) for C, V in DM.items()
                } for M, DM in DU.items()
            } for U, DU in AccessDict.items()
        }, indent=4)))

    metrics += [
        {
            "labels": {
                "sensor": "http.server.errors",
                "error": L
            },
            "value": V/Coef
        } for L, V in ErrorDict.items()
    ]
    if Debug:
        print("Error metrics: {}".format(json.dumps({
            L: "{:.1f}/{:.2f} = {:.6f}".format(V, Coef, V/Coef) for L, V in ErrorDict.items()
        }, indent=4)))

    TtlDict = {CertFile: TTLDays for CertFile, TTLDays in CertsTTL(NginxDir)}
    metrics += [
        {
            "labels": {
                "sensor": "certificateTtlDays",
                "file": F
            },
            "value": T
        } for F, T in TtlDict.items()
    ]
    if Debug:
        print("Cert TTL metrics: {}".format(json.dumps(TtlDict, indent=4)))

    try:
        File = open(JsonFile, "w+")
        File.write(json.dumps({"metrics": metrics}))
    except IOError:
        print("ERROR: cannot write json to '{}'".format(JsonFile))


if __name__ == "__main__":
    Main()
