#!/usr/bin/env python3

import os
import sys
import csv
import pprint
import collections
import datetime as dt
import solomon_api

def main():
    api = solomon_api.V2(os.environ["OAUTH_TOKEN"])
    url = '/projects/yabs/sensors/data'

    end = dt.datetime.now()
    begin = end - dt.timedelta(days=300)

    sensors = set(['timestamp'])
    result = collections.defaultdict(dict)
    while begin <= end:
        f = begin.strftime("%Y-%m-%dT%H:%M:%S.%fZ")
        t = (begin + dt.timedelta(days=7)).strftime("%Y-%m-%dT%H:%M:%S.%fZ")
        print("from={}, to={}".format(f, t))
        data = {
            'program': '{cluster="artmon_metric_proxy", service="artmon_metric_proxy", host="artmon-metric-pusher", sensor="fast_stat.yt_bigrt.*"}',
            'from': f,
            'to': t,
            'downsampling': {
                # "disabled": True
                'gridMillis': 5 * 60 * 1000
            }
        }

        metrics = api._post_json(url, data).get('vector', [])

        for m in metrics:
            sensor = m['timeseries']['labels']['sensor']
            sensors.add(sensor)
            values = m['timeseries'].get('values', [])

            if len(values) > 0:
                timestamps = m['timeseries'].get('timestamps')
                if len(values) != len(timestamps):
                    print("Unk")
                    sys.exit(1)
                for t, v in zip(timestamps, values):
                    result[t][sensor] = v

        begin += dt.timedelta(days=7)

    # pprint.pprint(result)

    with open('MONSUPPORT-337_5m.csv', 'w') as f:
        writer = csv.DictWriter(f, fieldnames=list(sensors))
        writer.writeheader()

        for r in result:
            row = result[r]
            row['timestamp'] = dt.datetime.fromtimestamp(r / 1000).strftime("%Y-%m-%dT%H:%M:%S.%fZ")

            writer.writerow(row)


if __name__ == '__main__':
    main()
