#!/usr/bin/env python3

import json
import sys
import time
import urllib.request, urllib.error, urllib.parse


def fetch(url):
    return json.load(urllib.request.urlopen(url))["sensors"]


def parse_labels(labels_str):
    labels = {}
    for label in filter(bool, labels_str.split("&")):
        key, value = label.split("=")
        labels[key] = value
    return labels


def match_labels(labels, pattern):
    for key, value in pattern.items():
        if key not in labels:
            return False
        if labels[key] != value:
            return False
    return True


def labels_to_sensor_name(labels):
    name = labels.get("sensor", "")
    if name:
        del labels["sensor"]
    tags = ", ".join(key + "=" + labels[key] for key in sorted(labels))
    return name + "{" + tags + "}"


def main(argv):
    if len(argv) != 3:
        print("Usage:", argv[0], "<url>", "<labels>", file=sys.stderr)
        sys.exit(1)

    url = argv[1]
    labels = parse_labels(argv[2])
    prev_value = None
    interval = 15  # seconds

    while True:
        started = time.time()

        sensors = fetch(url)
        for sensor in sensors:
            if match_labels(sensor["labels"], labels):
                if prev_value is not None:
                    print(time.strftime("%Y-%m-%d %H:%M:%S"), \
                            labels_to_sensor_name(sensor["labels"]), \
                            (sensor["value"] - prev_value) / interval)
                prev_value = sensor["value"]
                break

        elapsed = time.time() - started
        if elapsed < interval:
            time.sleep(interval - elapsed)


if __name__ == "__main__":
    try:
        main(sys.argv)
    except KeyboardInterrupt:
        pass
