#!/usr/bin/env python3

import argparse
import json
import os
import sys

import solomon_api

OAUTH_TOKEN_ENV_VAR = "OAUTH_TOKEN"


def get_internal_project_objects(client, object_name, project):
    result, object_items = [], []
    page, pagesCount = 0, 10500

    while page <= pagesCount:
        try:
            response = client._get_json("/projects/" + project + "/" + object_name + "?page=" + str(page) + "&pageSize=1000")
            pagesCount = response["page"]["pagesCount"]
        except Exception as e:
            print("Failed to load " + object_name + " in project " + project, e, file=sys.stderr)
            return result
        
        if 'items' in response:
            object_items.extend(response["items"])
        elif 'result' in response:
            object_items.extend(response["result"])
        
        page += 1
    
    print("Found total {} {} in project {}".format(len(object_items), object_name, project))

    for object_item in object_items:
        object_id = object_item["id"]
        try:
            obj = client._get_json("/projects/" + project + "/" + object_name + "/" + object_id)
            result.append(obj)
            print("Load " + object_id + " in " + project)
        except Exception as e:
            print("Failed to load " + object_id + ' in ' + project, e, file=sys.stderr)
    print()
    return result


def get_project(client, project):
    try:
        response = client._get_json("/projects/" + project)
        return [response]
    except Exception as e:
        print("Failed to load project " + project, e, file=sys.stderr)
        return []


def get_project_objects(client, project, object_name):
    print("Project", project)
    print()

    if object_name == 'projects':
        return get_project(client, project)

    return get_internal_project_objects(client, object_name, project)


def get_all_objects(client, object_name):
    result = []
    all_project_items = client._get_json("/projects")
    for cur_project_item in all_project_items:
        cur_project_id = cur_project_item["id"]
        project_objects = get_project_objects(client, cur_project_id, object_name)
        result.extend(project_objects)
    return result


def download_from_object_items(client, object_name, object_items):
    result = []
    for object_item in object_items:
        project_id = object_item["projectId"]
        object_id = object_item["id"]
        try:
            obj = client._get_json("/projects/" + project_id + "/" + object_name + "/" + object_id)
            result.append(obj)
            print("Load " + object_id + " in " + project_id)
        except Exception as e:
            print("Failed to load " + object_id + ' in ' + project_id, e, file=sys.stderr)
    return result


def get_all_objects_fast(client, object_name):
    result = []

    page = 0

    # little hack to support API v2 and internal API both
    v2_endpoint = client._endpoint
    internal_endpoint = client._endpoint.replace("v2", "internal")

    while True:
        print("Load %s page # %s..." % (object_name, page + 1))
        try:
            # little hack to support internal API
            client._endpoint = internal_endpoint
            response = client._get_json("/" + object_name + "?pageSize=1000&page=" + str(page))
        except Exception as e:
            print("Failed to load %s page #%s" % (object_name, str(page + 1)), e, file=sys.stderr)
            return result
        # little hack to use API v2 again
        client._endpoint = v2_endpoint

        object_items = response["result"]
        page_info = response["page"]
        pages_count = page_info["pagesCount"]

        page_objects = download_from_object_items(client, object_name, object_items)

        result.extend(page_objects)

        if page + 1 >= pages_count:
            return result

        page += 1


def main():
    if OAUTH_TOKEN_ENV_VAR not in os.environ:
        print("OAuth token isn't specified in", OAUTH_TOKEN_ENV_VAR, "env variable", file=sys.stderr)
        sys.exit(100)

    oauth_token = os.environ[OAUTH_TOKEN_ENV_VAR]

    parser = argparse.ArgumentParser()
    parser.add_argument("object", help="object name (projects, clusters, services, shards, graphs, dashboards, alerts, channels)")
    parser.add_argument("--project", default="", help="project id")
    parser.add_argument("--fast", action="store_true", help="increase downloading using internal API")
    parser.add_argument("--env", help="environment (PRODUCTION|PRESTABLE|TESTING|CLOUD_PREPROD|CLOUD_PROD)")

    args = parser.parse_args()

    object_name = args.object
    project = args.project if args.project is not None else ""
    is_fast = args.fast
    env = args.env if args.env is not None else "PRODUCTION"

    client = solomon_api.V2(oauth_token, env)

    if project == "":
        if is_fast:
            result = get_all_objects_fast(client, object_name)
        else:
            result = get_all_objects(client, object_name)
    else:
        result = get_project_objects(client, project, object_name)

    filename = ("all" if project == "" else project) + "_" + object_name + ".json"
    with open(filename, "w") as file:
        json.dump(result, file, indent = 4)


if __name__ == "__main__":
    main()
