#!/usr/bin/env python3

import getpass
import json
import os
import psutil
import sys
import time
import urllib.error
import urllib.request


SOLOMON_API = "https://solomon.yandex.net/api/v2"


def gauge(name, value, labels={}):
    return {'kind': 'GAUGE', 'name': name, 'labels': labels, 'value': value}


def counter(name, value, labels={}):
    return {'kind': 'COUNTER', 'name': name, 'labels': labels, 'value': value}


def get_load_metrics():
    load = psutil.getloadavg()
    return [
        gauge('load_avg_1m', load[0]),
        gauge('load_avg_5m', load[1]),
        gauge('load_avg_15m', load[2]),
    ]


def get_cpu_metrics():
    stats = psutil.cpu_stats()
    times = psutil.cpu_times()
    times_percent = psutil.cpu_times_percent()
    return [
        gauge('cpu_percent', psutil.cpu_percent()),
        gauge('cpu_count', psutil.cpu_count()),
        gauge('cpu_freq', psutil.cpu_freq().current),
        gauge('cpu_stats', stats.ctx_switches, {'stat': 'ctx_switches'}),
        gauge('cpu_stats', stats.interrupts, {'stat': 'interrupts'}),
        gauge('cpu_stats', stats.soft_interrupts, {'stat': 'soft_interrupts'}),
        gauge('cpu_stats', stats.syscalls, {'stat': 'syscalls'}),
        gauge('cpu_times', times.user, {'time': 'user'}),
        gauge('cpu_times', times.nice, {'time': 'nice'}),
        gauge('cpu_times', times.system, {'time': 'system'}),
        gauge('cpu_times', times.idle, {'time': 'idle'}),
        gauge('cpu_times_precent', times_percent.user, {'time': 'user'}),
        gauge('cpu_times_precent', times_percent.nice, {'time': 'nice'}),
        gauge('cpu_times_precent', times_percent.system, {'time': 'system'}),
        gauge('cpu_times_precent', times_percent.idle, {'time': 'idle'}),
    ]


def get_mem_metrics():
    mem_virtual = psutil.virtual_memory()
    mem_swap = psutil.swap_memory()
    return [
        gauge('mem_virtual', mem_virtual.total),
        gauge('mem_virtual', mem_virtual.available, {'type': 'available'}),
        gauge('mem_virtual', mem_virtual.used, {'type': 'used'}),
        gauge('mem_virtual', mem_virtual.free, {'type': 'free'}),
        gauge('mem_virtual', mem_virtual.active, {'type': 'active'}),
        gauge('mem_virtual', mem_virtual.inactive, {'type': 'inactive'}),
        gauge('mem_virtual', mem_virtual.wired, {'type': 'wired'}),
        gauge('mem_swap', mem_swap.total),
        gauge('mem_swap', mem_swap.used, {'type': 'used'}),
        gauge('mem_swap', mem_swap.free, {'type': 'free'}),
        counter('mem_swap', mem_swap.sin, {'type': 'sin'}),
        counter('mem_swap', mem_swap.sout, {'type': 'sout'}),
    ]


def get_disk_metrics():
    io = psutil.disk_io_counters()
    return [
        counter('disk_io', io.read_count, {'type': 'read_count'}),
        counter('disk_io', io.write_count, {'type': 'write_count'}),
        counter('disk_io', io.read_bytes, {'type': 'read_bytes'}),
        counter('disk_io', io.write_bytes, {'type': 'write_bytes'}),
        counter('disk_io', io.read_time, {'type': 'read_time'}),
        counter('disk_io', io.write_time, {'type': 'write_time'}),
    ]


def get_net_metrics():
    io = psutil.net_io_counters()
    return [
        counter('net_io', io.bytes_sent, {'type': 'bytes_sent'}),
        counter('net_io', io.bytes_recv, {'type': 'bytes_recv'}),
        counter('net_io', io.packets_sent, {'type': 'packets_sent'}),
        counter('net_io', io.packets_recv, {'type': 'packets_recv'}),
        counter('net_io', io.errin, {'type': 'errin'}),
        counter('net_io', io.errout, {'type': 'errout'}),
        counter('net_io', io.dropin, {'type': 'dropin'}),
        counter('net_io', io.dropout, {'type': 'dropout'}),
    ]


def push_metrics(project, cluster, service, token, metrics):
    url = SOLOMON_API + "/push?project=" + project + \
        "&cluster=" + cluster + "&service=" + service
    data = json.dumps(metrics).encode('utf8')
    headers = {
        'Authorization': 'OAuth ' + token,
        'Content-Type': 'application/json',
        'Accept': 'application/json',
    }
    try:
        req = urllib.request.Request(url, data=data, headers=headers)
        resp = urllib.request.urlopen(req)
        return json.load(resp)
    except urllib.error.HTTPError as e:
        print(e.code, e.reason)
        print(e.read().decode('utf8'))
        sys.exit(1)


def main():
    token = os.environ.get("SOLOMON_TOKEN", None)
    if not token:
        print("No SOLOMON_TOKEN environment variable.")
        print("Please go to https://oauth.yandex-team.ru/authorize?response_type=token&client_id=1c0c37b3488143ff8ce570adb66b9dfa")
        print("and do:")
        print("  $ export SOLOMON_TOKEN=AQAD-*************************")
        sys.exit(1)

    project = "notebook"
    cluster = getpass.getuser()
    service = "system"

    while True:
        start_ts = time.time()
        ts = int(start_ts)
        ts -= ts % 5

        all_metrics = [
            get_load_metrics(),
            get_cpu_metrics(),
            get_mem_metrics(),
            get_disk_metrics(),
            get_net_metrics(),
        ]

        # flatten lists
        metrics = [item for sublist in all_metrics for item in sublist]
        req = {'ts': ts, 'metrics': metrics}
        resp = push_metrics(project, cluster, service, token, req)

        time_str = time.strftime("%H:%M:%S", time.localtime(ts))
        print("%s [%s] pushed %d metrics" %
              (time_str, resp['status'], resp['sensorsProcessed']))

        elapsed_ts = time.time() - start_ts
        sleep_ts = 5.0 - elapsed_ts
        if sleep_ts > 0:
            time.sleep(sleep_ts)


if __name__ == "__main__":
    main()
