#!/usr/bin/env python3

import json
import os
import sys
import logging

from collections import defaultdict
from datetime import datetime, timedelta
from dateutil.parser import parse
from enum import Enum
from urllib.error import HTTPError
from urllib.parse import quote
from urllib.request import Request, urlopen


LOG_FORMAT = "%(asctime)-15s %(name)s:%(lineno)-4s %(levelname)-8s %(message)s"
logging.basicConfig(format=LOG_FORMAT, level=logging.INFO)
log = logging.getLogger('notify.py')

GROUP_CHATS = {
        "solomon-ops": 18446743072607889496,
        "yasm-alerts": 18446743072356338369,
}

PEOPLE_CHATS = {
        "gordiychuk": 294181914,
        "guschin": 23902280,
        "knuzhdin": 148365833,
        "ivanzhukov": 65687432,
        "uranix": 271780368,
        "samarius": 138835032,
        "hpple": 107174949,
        "alextrushkin": 474809837,
        "ayasu": 0,
        "amaramchin": 0,
        "kgershov": 884093384,
}

DUTY_PERSON_MESSAGE = """
SRE дежурный: {name}

email: {email}
tel: {phone}

@{telegram}
"""

SUPPORT_PERSON_MESSAGE = """
За поддержку пользователей отвечает: {name}

email: {email}
tel: {phone}

@{telegram}
"""


class DutyTypes(Enum):
    SOLOMON_SRE = 'Solomon SRE'
    SOLOMON_SUPPORT = 'Solomon Users Support'


SERVICE_IDS = {
    DutyTypes.SOLOMON_SRE: 700,  # abc/services/solomon
    DutyTypes.SOLOMON_SUPPORT: 700,  # abc/services/solomon
}

SCHEDULE_NAMES = {
    DutyTypes.SOLOMON_SRE: 'SRE',
    DutyTypes.SOLOMON_SUPPORT: 'Поддержка пользователей и инцидент-менеджмент',
}

WIKIS = {
    DutyTypes.SOLOMON_SRE: '\* https://wiki.yandex-team.ru/solomon/dev/duty\n'
                           '\* https://wiki.yandex-team.ru/solomon/dev/release\n'
                           '\* https://wiki.yandex-team.ru/golovan/dev/duty',
    DutyTypes.SOLOMON_SUPPORT: '\* https://wiki.yandex-team.ru/solomon/support',
}

SRE_CHATS = set([
    '\* [Monitoring Emergency](https://t.me/joinchat/AAiKMz8nRz-hIw1ncjZwxw)',
    '\* [Solomon Ops](https://t.me/joinchat/QaoHqNO0mYF6Xu6I)',
    '\* [yasm-alerts](https://t.me/joinchat/EYjcGlCoZT-CEMWnMI50ig)',
    '\* [Cloud PROD Support](https://t.me/joinchat/AXmZSke_Y2Hv6HTStRCPMg)',
    '\* [Cloud PRE & TESTING Support](https://t.me/joinchat/AXmZSkKKXwhyKH-lJyOLtw)',
    '\* [CloudIL (ISRAEL) support](https://t.me/+Ah4l1V-zwbEzMjMy)',
])

# TODO: pass links in markdown
CHATS_TO_MONITOR = {
    DutyTypes.SOLOMON_SRE: SRE_CHATS,
    DutyTypes.SOLOMON_SUPPORT: set([
        '\* [Monitoring Community](https://t.me/joinchat/AWy4SEK1sYQoq4Ign2oINQ)',
    ]),
}

MAILING_LISTS_TO_MONITOR = {
    DutyTypes.SOLOMON_SUPPORT: set([
        '\* [solomon@](https://ml.yandex-team.ru/lists/solomon)',
    ]),
}

QUEUES_TO_MONITOR = {
    DutyTypes.SOLOMON_SUPPORT: set([
        '\* [MONSUPPORT](https://st.yandex-team.ru/MONSUPPORT)',
        '\* [MONITORINGREQ](https://st.yandex-team.ru/MONITORINGREQ)',
        '\* [CLOUDSUPPORT](https://st.yandex-team.ru/CLOUDSUPPORT)',
        '\* [GOLOVANSUPPORT](https://st.yandex-team.ru/GOLOVANSUPPORT)',
    ]),
}


PERSONAL_MESSAGE = """
Ты сегодня дежурный сервисов: {duties}.

{chats}{mailing_lists}{queues}

Подробнее см.:{wikis}
"""

NEXT_PERSONAL_MESSAGE = """
Напоминаю, что завтра ты будешь дежурным в: {duties}.
Подробнее см.:{wikis}
"""

STANDUP_MESSAGE = """
Приходи на стендап
https://yandex.zoom.us/j/147954717?pwd=a0FRQm45RWhYeHFoQUNyaFFxRkJkZz09
"""

WEEKLY_MEETING_MESSAGE = """
Приходи на еженедельную встречу
https://yandex.zoom.us/j/845718129?pwd=RW1wZmo3emhsUGNMRUd3OFJReUQ4UT09
"""


class HttpClient:
    def __init__(self, url, token=None):
        self._url = url
        self._token = token

    def get_json(self, path):
        log.debug("GET %s", path)
        req = Request(self._url + path)
        if self._token:
            req.add_header("Authorization", "OAuth " + self._token)
        try:
            resp = urlopen(req)
            return json.loads(resp.read().decode("utf-8"))
        except HTTPError as e:
            log.error('HTTP error: %s %s', e.code, e.reason)
            log.error('%s', e.read().decode('utf8'))
            raise e


def shift_is_active(shift, timestamp):
   ts_start = parse(shift["start_datetime"]).timestamp()
   ts_end = parse(shift["end_datetime"]).timestamp()
   return ts_start <= timestamp < ts_end


class AbcClient(HttpClient):
    BASE_URL = "https://abc-back.yandex-team.ru/api/v4"

    def __init__(self, token):
        super(AbcClient, self).__init__(self.BASE_URL, token)

    def get_shift(self, duty_type, date):
        service_id = SERVICE_IDS[duty_type]
        schedule_name = SCHEDULE_NAMES[duty_type]

        path = "/duty/shifts/?service={service_id}" \
                "&date_from={date}&date_to={date}" \
                "&fields=schedule.id,schedule.name," \
                "person.id,person.login," \
                "replaces.id,replaces.person.id,replaces.person.login"
        resp = self.get_json(path.format(
            service_id=service_id,
            date=date.strftime("%Y-%m-%d")))

        for shift in resp["results"]:
            log.debug('Got shift %s', json.dumps(shift, indent=2))
            if shift["schedule"]["name"] == schedule_name and shift_is_active(shift, date.timestamp()):
                for replace in shift.get("replaces", []):
                    if shift_is_active(replace, date.timestamp()):
                        return replace["person"]
                return shift["person"]
        raise RuntimeError("cannot find shift with name '" + schedule_name + "'")


class StaffClient(HttpClient):
    BASE_URL = "https://staff-api.yandex-team.ru/v3"

    def __init__(self, token):
        super(StaffClient, self).__init__(self.BASE_URL, token)

    def get_person(self, login):
        path = "/persons?_one=1&login={login}&_fields={fields}".format(
                login=login,
                fields="phones,accounts,name,work_email,work_phone")
        return Person(login, self.get_json(path))


class TelegramClient(HttpClient):
    BASE_URL = "https://api.telegram.org/bot"

    def __init__(self, token):
        super(TelegramClient, self).__init__(self.BASE_URL + token)

    def send_message(self, chat_id, text):
        format_options = "&parse_mode=Markdown&disable_web_page_preview=true"
        self.get_json("/sendMessage?chat_id={chat_id}&text={text}{format_options}".format(
            chat_id=chat_id,
            text=quote(text),
            format_options=format_options))


class Person:
    def __init__(self, login, data):
        self._login = login
        self._data = data

    def __hash__(self):
        return hash(self._login)

    def __eq__(self, other):
        return self._login == other._login

    def login(self):
        return self._login

    def name(self):
        name = self._data["name"]
        return name["first"]["ru"] + " " + name["last"]["ru"]

    def email(self):
        return self._data["work_email"]

    def phone(self):
        for p in self._data["phones"]:
            if p.get("is_main", False):
                return p["number"]
        return None

    def telegram(self):
        for a in self._data["accounts"]:
            if a.get("type", "") == "telegram" and not a.get("private", False):
                return a["value"]
        return None


def send_chat_message(tg, group_id, template, person):
    message = template.format(
        name=person.name(),
        email=person.email(),
        phone=person.phone(),
        telegram=person.telegram())
    tg.send_message(group_id, message)


def construct_sources_to_monitor(duties, SOURCES):
    s = ''

    for duty in duties:
        if duty in SOURCES:
            s += '- ' + duty.value + ':\n'
            s += '\n'.join('    ' + tmp for tmp in SOURCES[duty]) + '\n'

    return s


def construct_msg_parts(duties):
    if datetime.today().weekday() != 0:
        duties = [
            duty for duty in duties
            if duty not in [DutyTypes.SOLOMON_SUPPORT]
        ]

    if len(duties) == 0:
        return None

    wikis = ''.join('\n' + WIKIS[duty_type] for duty_type in duties)

    # TODO: if chats/queues/etc overlap, maybe it'd be better to left only unique values
    chats = construct_sources_to_monitor(duties, CHATS_TO_MONITOR)
    mailing_lists = construct_sources_to_monitor(duties, MAILING_LISTS_TO_MONITOR)
    queues = construct_sources_to_monitor(duties, QUEUES_TO_MONITOR)

    return {
        'duties': ', '.join(duty.value for duty in duties),
        'mailing_lists': mailing_lists,
        'wikis': wikis,
        'queues': queues,
        'chats': chats,
    }


def prepend_prefix(s, prefix):
    if len(s) == 0:
        return s

    return '\n' + prefix + ':\n' + s + '\n'


def send_personal_message(tg, person, duties):
    chat_id = PEOPLE_CHATS.get(person.login(), None)
    msg_parts = construct_msg_parts(duties)

    if not msg_parts:
        return

    chats = prepend_prefix(msg_parts['chats'], 'За какими чатами следить')
    mailing_lists = prepend_prefix(msg_parts['mailing_lists'], 'Какие рассылки читать')
    queues = prepend_prefix(msg_parts['queues'], 'За какими очередями следить')

    if chat_id and msg_parts:
        msg = PERSONAL_MESSAGE.format(
            duties=msg_parts['duties'],
            wikis=msg_parts['wikis'],
            mailing_lists=mailing_lists,
            queues=queues,
            chats=chats,
        )
        tg.send_message(chat_id, msg)


def remind_about_duty(tg, person, duties):
    chat_id = PEOPLE_CHATS.get(person.login(), None)
    msg_parts = construct_msg_parts(duties)

    if chat_id and msg_parts:
        msg = NEXT_PERSONAL_MESSAGE.format(
            duties=msg_parts['duties'],
            wikis=msg_parts['wikis'],
        )
        tg.send_message(chat_id, msg)


def print_person(person, duties):
    print("В {duties} дежурит: {person_name}".format(
        duties=[duty.value for duty in duties],
        person_name=person.name()
    ))
    print("  email:", person.email())
    phone = person.phone()
    if phone:
        print("    tel:", phone)
    telegram = person.telegram()
    if telegram:
        print("     tg: https://telegram.me/" + telegram)


def get_assignments(abc, staff, date):
    person_to_duties = defaultdict(list)
    duty_to_person = {}

    for duty_type in DutyTypes:
        try:
            person = staff.get_person(abc.get_shift(duty_type, date)['login'])
            person_to_duties[person].append(duty_type)
            duty_to_person[duty_type] = person
        except Exception as err:
            log.error("failed to get duty info for %s: %s", str(duty_type), str(err))

    return person_to_duties, duty_to_person


# For testing. Pass an instance of this class in a send_*() function you want to test
class FakePerson():
    def login(self):
        return os.environ.get('TEST_LOGIN', 'your_staff_login_here')


def main():
    cmd = "print"

    if len(sys.argv) > 1:
        cmd = sys.argv[1]

    conf = json.load(open("/etc/solomon/duty.conf", "r"))

    abc = AbcClient(conf["robot_oauth_token"])
    staff = StaffClient(conf["robot_oauth_token"])
    tg = TelegramClient(conf["telegram_bot_token"])

    if cmd == "standup":
        for login, chat_id in PEOPLE_CHATS.items():
            try:
                tg.send_message(chat_id, STANDUP_MESSAGE)
            except:
                # ignore error, logged inside send_message
                pass
        return

    if cmd == "weekly":
        for login, chat_id in PEOPLE_CHATS.items():
            try:
                tg.send_message(chat_id, WEEKLY_MEETING_MESSAGE)
            except:
                # ignore error, logged inside send_message
                pass
        return

    today = datetime.now()
    person_to_duties, duty_to_person = get_assignments(abc, staff, today)

    if cmd == "duty":
        is_testing = os.environ.get('BOT_TESTING', '') == 'true'

        if not is_testing:
            send_chat_message(
                tg, GROUP_CHATS['solomon-ops'],
                DUTY_PERSON_MESSAGE, duty_to_person[DutyTypes.SOLOMON_SRE]
            )

            if today.weekday() == 0:
                send_chat_message(
                    tg, GROUP_CHATS['solomon-ops'],
                    SUPPORT_PERSON_MESSAGE, duty_to_person[DutyTypes.SOLOMON_SUPPORT]
                )
                send_chat_message(
                    tg, GROUP_CHATS['yasm-alerts'],
                    DUTY_PERSON_MESSAGE, duty_to_person[DutyTypes.SOLOMON_SRE]
                )

        for person, duties in person_to_duties.items():
            person = FakePerson() if is_testing else person
            send_personal_message(tg, person, duties)

        tomorrow = today + timedelta(days=1)
        tomorrow_person_to_duties, _ = get_assignments(abc, staff, tomorrow)

        for person, duties in tomorrow_person_to_duties.items():
            person = FakePerson() if is_testing else person
            remind_about_duty(tg, person, duties)
    else:
        for person, duties in person_to_duties.items():
            print_person(person, duties)


if __name__ == "__main__":
    main()
