#!/usr/bin/env python3

import sys
import urllib.request, urllib.parse, urllib.error

from lxml import etree
from collections import defaultdict


MAN = "kikimr0340.search.yandex.net"
SAS = "kikimr0101.search.yandex.net"

MAN_CLUSTER = "kikimr_solomon_man"
SAS_CLUSTER = "kikimr_solomon_sas"

SOLOMON_URL = "https://solomon.yandex-team.ru"
SOLOMON_URL_PARAMS = {
        'project': 'solomon',
        'cluster': '<CLUSTER>',
        'service': 'salmon_stockpile',
        'stockpile_shard_id': '<IDS>',
        'path': '/shard_act_millis',
        'checks': '-',
        'stack': 'false',
        'transform': 'moving_average',
        'secondaryGraphMode': 'none',
        'filter': 'top',
        'filterLimit': '10',
        'filterBy': 'max',
        'b': '1h',
        'graph': 'auto',
}

class Format:
    def header(self):
        pass

    def graph(self, host, url_path, ids):
        pass

    def footer(self):
        pass


class HtmlFormat(Format):
    def header(self):
        print("<html><body><ul>")

    def graph(self, host, url_path, ids):
        print("<li><a href='%s/?%s'>%s (%d)</a></li>" % (
                SOLOMON_URL, url_path, host, len(ids)))

    def footer(self):
        print("</ul></body></html>")


class SolomonFormat(Format):
    def graph(self, host, url_path, ids):
        print("rows {")
        print("  elements {")
        print("    url: \"/?%s\"" % url_path)
        print("    label: \"%s (%d)\"" % (host, len(ids)))
        print("  }")
        print("}")


def main(argv):
    if len(argv) != 3:
        print("Usage: %s {man|sas} {html|solomon}" % argv[0])
        sys.exit(1)

    stockpile_host = SAS if argv[1] == "sas" else MAN
    SOLOMON_URL_PARAMS['cluster'] = SAS_CLUSTER if argv[1] == "sas" else MAN_CLUSTER
    f = HtmlFormat() if argv[2] == "html" else SolomonFormat()

    url = urllib.request.urlopen("http://%s:8606/stockpile-shards" % stockpile_host)
    dom = etree.HTML(url.read())

    table = dom.find("body/div/table")
    ids = table.xpath('tr/td[1]/a/text()')
    hosts = table.xpath('tr/td[2]/text()')

    shards = defaultdict(list)
    for id, host in zip(ids, hosts):
        shards[host].append(id)


    f.header()
    for host, ids in shards.items():
        SOLOMON_URL_PARAMS["stockpile_shard_id"] = "|".join(ids)
        url_path = urllib.parse.urlencode(SOLOMON_URL_PARAMS)
        f.graph(host, url_path, ids)
    f.footer()


if __name__ == "__main__":
    main(sys.argv)

