# -*- coding: utf-8 -*-

from http.server import HTTPServer, BaseHTTPRequestHandler
import solomon.scripts.quotas_helper.bootstrap as bootstrap
import solomon.scripts.quotas_helper.metrics as metrics
import solomon.scripts.quotas_helper.unistat as unistat
from library.python.monlib.metric_registry import MetricRegistry, HistogramType
#from urllib import parse
from urllib.parse import quote, quote_plus, unquote, unquote_plus, urlparse, parse_qsl, urlencode
import os
import re
import requests
import socket
import sys
import datetime
import time
import json
import random

HOST_NAME = os.getenv("HOST", "::")
PORT_NUMBER = os.getenv("PORT", 8080)

supporters = ["andreyst", "guschin", "samarius"]
supporters_idx = random.randint(0, len(supporters) - 1)

# Override staff gaps
# Day present in set means "NOT ABSENT on this day EVEN if staff says otherwise"
# Use e.g. for conference (which does not have "will work" flag)
absence_overrides = {
    "andreyst": set([
        datetime.date(2021, 10, 26), # Dash conference
        datetime.date(2021, 10, 27), # Dash conference
        datetime.date(2021, 11, 9), # ObservabilityCON
        datetime.date(2021, 11, 10), # ObservabilityCON
    ]),
    "guschin": set([
    ]),
    "samarius": set([
    ]),
}

def process_ticket(ticket):
    if 'MONITORINGREQ' in ticket:
        # print("Processing MONITORINGREQ")
        assign_ticket(ticket)
        process_monitoringreq_ticket(ticket)
    elif 'MONSUPPORT' in ticket:
        # print("Processing MONSUPPORT")
        assign_ticket(ticket)
    elif 'GOLOVANSUPPORT' in ticket:
        # print("Processing GOLOVANSUPPORT")
        assign_ticket(ticket)
    else:
        print("Nothing can do for ticket {}".format(ticket))

def assign_ticket(ticket):
    global supporters_idx

    st_token = os.environ["ST_TOKEN"]
    st_headers = {"Authorization": "OAuth {token}".format(token=st_token)}

    supporter = None
    reason = "primary"
    if ticket.startswith("MONITORINGREQ"):
        print("Looking for primary supporter")
        reason = "primary"
        supporter = find_primary_supporter()

    if not supporter:
        print("Looking for next available supporter")
        reason = "next available"
        supporter = find_available_supporter()

    if not supporter:
        print(f"ERROR: cannot assign support ticket {ticket} due to cannot find {reason} supporter: no schedule/bad schedule slug/everyone is unavailable/...")
        return

    data = {"assignee": supporter}
    res = requests.patch("https://st-api.yandex-team.ru/v2/issues/{ticket}".format(ticket=ticket), json=data, headers=st_headers)

    print(f"Ticket https://st.yandex-team.ru/{ticket} assigned to {reason}: {supporter}@")


def find_primary_supporter():
    staff_token = os.environ["STAFF_TOKEN"]
    staff_headers = {"Authorization": "OAuth {token}".format(token=staff_token)}

    res = requests.get("https://abc-back.yandex-team.ru/api/v4/duty/on_duty/?service=700&schedule__slug=support", headers=staff_headers).json()
    person = res[0].get("person", {})
    supporter = person.get("login", None)

    if not supporter:
        return None

    gaps = get_gaps([supporter])
    if is_absent(supporter, gaps[supporter]):
        print(f"Primary supporter {supporter} unavailable because of staff gap: {json.dumps(gaps[supporter], ensure_ascii=False)}")
        return None

    return supporter


def find_available_supporter():
    global supporters_idx

    gaps = get_gaps(supporters)

    start_idx = supporters_idx
    while True:
        supporter = supporters[supporters_idx]

        supporter_available = True
        if is_absent(supporter, gaps[supporter]):
            print(f"Skipping {supporter} because of staff gap: {json.dumps(gaps[supporter], ensure_ascii=False)}")
            supporter_available = False

        supporters_idx += 1
        if len(supporters) == supporters_idx:
            supporters_idx = 0

        if supporter_available:
            return supporter

        if supporters_idx == start_idx:
            return None


def absence_overrided(supporter):
    today = datetime.datetime.now().date()
    return supporter in absence_overrides and today in absence_overrides[supporter]


def is_absent(supporter, gaps):
    if absence_overrided(supporter):
        return False

    for gap in gaps:
        if (gap["workflow"] not in ["duty", "remote_work", "office_work"]) and ("work_in_absence" not in gap or not gap["work_in_absence"]):
            return True

    return False


def get_gaps(logins):
    staff_token = os.environ["STAFF_TOKEN"]
    staff_headers = {"Authorization": "OAuth {token}".format(token=staff_token)}
    logins_param = "&".join(["l={}".format(x) for x in logins])

    res = requests.get(f"https://staff.yandex-team.ru/gap-api/api/export_gaps?{logins_param}", headers=staff_headers)
    res = res.json()

    gaps = {}

    for login in logins:
        gaps[login] = res["persons"].get(login, [])

    return gaps


def process_monitoringreq_ticket(ticket):
    token = os.environ["ST_TOKEN"]

    headers = {
        "Authorization": "OAuth {token}".format(token=token)
    }

    res_raw = requests.get("https://st-api.yandex-team.ru/v2/issues/{ticket}".format(ticket=ticket), headers=headers)
    res = res_raw.json()

    token = os.environ["ST_TOKEN"]
    st_headers = {
        "Authorization": "OAuth {token}".format(token=token),
    }

    desc = res["description"]
    components = res.get("components", [])
    if components:
        for c in components:
            if c["id"] == "96508":

                from_dash_match = re.search("solomon.cloud-preprod.yandex-team.ru/admin/serviceProviders/([a-zA-Z0-9_-]+)/serviceDashboards/([a-zA-Z0-9_-]+)", desc)
                from_dash = from_dash_match.group(2)
                to_dash_match = re.search("solomon.cloud.yandex-team.ru/admin/serviceProviders/([a-zA-Z0-9_-]+)/serviceDashboards/([a-zA-Z0-9_-]+)", desc)
                to_dash = to_dash_match.group(2)

                print("Service Dasboard IDs found: cloud_pre://{} cloud-prod://{}".format(from_dash, to_dash))
                commands = "cd ~/arcadia/solomon/tools/service-dashboard && ya make -j16 && ./service-dashboard copy cloud_pre://{} cloud_prod://{} --ticket {}".format(from_dash, to_dash, ticket)

                result = re.search("Идентификатор проекта:\n([a-zA-Z0-9_-]+)\n", desc, flags=re.MULTILINE)
                data = {
                  "text": "\n<{{Команды для дежурного:\n%%{commands}%%}}>".format(commands=commands)
                }


                res = requests.post("https://st-api.yandex-team.ru/v2/issues/{ticket}/comments".format(ticket=ticket), json=data, headers=st_headers)

            if c["id"] == "76679":

                matches = []

                selectors_iter = re.finditer(r"https://(?P<installation>solomon.yandex-team.ru|solomon.cloud.yandex-team.ru)/admin/projects/(?P<project>[^/]+)/(?:metrics|sensors)\?selectors=(?P<selectors>[^ \n\r]+)", desc)
                for s in selectors_iter:
                    matches.append({
                        "installation": s.group("installation"),
                        "project": s.group("project"),
                        "selectors": unquote_plus(s.group("selectors"))
                    })

                alt_selectors_iter = re.finditer(r"https://(?P<installation>solomon.yandex-team.ru|solomon.cloud.yandex-team.ru)/\?(?P<full_selectors>[^ \n\r]+)", desc)
                for s in alt_selectors_iter:
                    full_selectors = s.group("full_selectors")
                    full_selectors_parsed = parse_qsl(full_selectors)

                    project = None
                    rm_idx = None
                    for i in range(len(full_selectors_parsed)):
                        if full_selectors_parsed[i][0] == "project":
                            project = full_selectors_parsed[i][1]
                            rm_idx = i
                        if full_selectors_parsed[i][0].startswith("l."):
                            full_selectors_parsed[i] = (full_selectors_parsed[i][0][2:], full_selectors_parsed[i][1])

                    if rm_idx is None:
                        continue

                    del full_selectors_parsed[rm_idx]

                    matches.append({
                        "installation": s.group("installation"),
                        "project": project,
                        "selectors": ", ".join(f"{x[0]}=\"{x[1]}\"" for x in full_selectors_parsed)
                    })

                commands = ""

                for match in matches:
                    installation = match["installation"]
                    project = match["project"]
                    selectors = match["selectors"]
                    data_str = json.dumps({
                        "description": ticket,
                        "selectors": selectors,
                        }, ensure_ascii=False)

                    commands += ("(operationId=$(curl --fail -X POST -H 'Content-Type: application/json' -H \"Authorization: OAuth $SOLOMON_TOKEN\" -d '{data_str}' https://{installation}/api/v2/projects/{project}/sensors/deletion | jq -r \".operationId\"); if [ -z \"$operationId\" ]; then echo -e \"\\033[0;31mFAIL\\033[0m Cannot create deletion operation with selectors {selectors}\"; exit; fi; echo \"Deleting metrics by selectors {selectors}, operationId=$operationId...\"; loop=true; while [ \"$loop\" = \"true\" ]; do status=$(curl --silent --fail -H \"Authorization: OAuth $SOLOMON_TOKEN\" https://{installation}/api/v2/projects/{project}/sensors/deletion/$operationId | jq -r '.status'); progress=$(curl --silent --fail -H \"Authorization: OAuth $SOLOMON_TOKEN\" https://{installation}/api/v2/projects/{project}/sensors/deletion/$operationId | jq -r '.progressPercentage'); echo \"$status ($progress%)...\"; if [ \"$status\" == \"WAITING_FOR_PERMANENT_DELETION\" ]; then loop=false; fi; sleep 5; done; estimatedMetricsCount=$(curl --silent --fail -H \"Authorization: OAuth $SOLOMON_TOKEN\" https://{installation}/api/v2/projects/{project}/sensors/deletion/$operationId | jq -r '.estimatedMetricsCount'); permanentDeletionAt=$(curl --silent --fail -H \"Authorization: OAuth $SOLOMON_TOKEN\" https://{installation}/api/v2/projects/{project}/sensors/deletion/$operationId | jq -r '.permanentDeletionAt'); echo -e \"\\033[0;32mOK\\033[0m Deleted $estimatedMetricsCount metrics with selectors {selectors}, permanent deletion at: $permanentDeletionAt\");\n\n\n").format(data_str=data_str, project=project, selectors=selectors.replace('"', '\\"'), installation=installation)

                commands += ("curl --fail -XPOST -H \"Authorization: OAuth $ST_TOKEN\" -H 'Content-Type: application/json' -d '{{\"resolution\": \"fixed\"}}' 'https://st-api.yandex-team.ru/v2/issues/{ticket}/transitions/closed/_execute' | jq 'if type==\"object\" and has(\"statusCode\") then .statusCode else \"success\" end' | grep success ; test $? -eq 0 && echo -e '\\n\\033[0;32mOK\\033[0m closing ticket' || echo -e '\\n\\033[0;31mFAIL\\033[0m closing ticket, retry curl manually to see error message (ticket already closed?)';\n\n".format(ticket=ticket))

                data = {
                  "text": "\n<{{Команды для дежурного:\n%%\n{commands}%%}}>".format(commands=commands)
                }


                res = requests.post("https://st-api.yandex-team.ru/v2/issues/{ticket}/comments".format(ticket=ticket), json=data, headers=st_headers)


    result = re.search("Идентификатор проекта:\n([a-zA-Z0-9._-]+)\n", desc, flags=re.MULTILINE)
    if result:
        project = result.group(1)

        result = re.search("Инсталляция:\n(solomon.(?:cloud.)?yandex-team.ru)\n", desc, flags=re.MULTILINE)
        if result:
            installation = result.group(1)

            result = re.findall("Тип квоты - (.*?)\nИдентификатор шарда - ([a-zA-Z0-9._*-]+?)\nТребуемое значение квоты - (.*?)\n", desc, flags=re.MULTILINE)
            shards_updates = {}
            for request in result:
                shards_updates = process_request(project, request[0], request[1], int(request[2]), shards_updates)

            process_updates(installation, project, ticket, shards_updates)


def process_request(project, quota_type, shard, quota_needed, shards_updates):
    if shard not in shards_updates:
        shards_updates[shard] = {}

    if quota_type == "Количество метрик в шарде":
        quota_key = "maxFileSensors"
    elif quota_type == "Количество метрик в ответе":
        quota_key = "maxSensorsPerUrl"
    elif quota_type == "Количество memonly-метрик":
        quota_key = "maxMemSensors"
    elif quota_type == "Размер ответа (в МБ)":
        quota_key = "maxResponseSizeMb"
    else:
        raise ValueError("Unknown quota type {quota_type}, fix script".format(quota_type=quota_type))

    shards_updates[shard][quota_key] = quota_needed

    return shards_updates


def process_updates(installation, project, ticket, shards_updates):

    token = os.environ["SOLOMON_TOKEN"]

    headers = {
        "Authorization": "OAuth {token}".format(token=token),
    }

    comment = ""
    commands = ""

    isCloud = installation == "solomon.cloud.yandex-team.ru"

    for shard, updates in shards_updates.items():
        res_raw = requests.get("https://{installation}/api/v2/projects/{project}/shards/{shard}".format(installation=installation, project=project, shard=shard), headers=headers)
        res = res_raw.json()
        if "code" in res:
          comment += f"Шард %%{shard}%% в проекте %%{project}%% найти не удалось.\n"
          print(f"ERROR: Error getting shard {shard} from project {project}: {res}")
          continue

        data = res

        comment += "((https://{installation}/admin/projects/{project}/shards/{shard} {shard}))\n".format(installation=installation, shard=shard, project=project)

        hasFileLimits = False
        hasMemLimits = False

        for quota_key, quota_needed in updates.items():
            quota_now = res["quotas"][quota_key]
            diff = quota_needed - quota_now
            data["quotas"][quota_key] = quota_needed
            comment += "{quota_key}: сейчас {quota_now:,}, надо {quota_needed:,}, diff {diff:,}\n".format(quota_key=quota_key, quota_now=quota_now, quota_needed=quota_needed, diff=diff)
            if quota_key == "maxFileSensors":
              hasFileLimits = True
            if quota_key == "maxMemSensors":
              hasMemLimits = True

        if hasFileLimits and not isCloud:
          comment += "\nfileSensorsLimits:\n{{{{iframe height='200' width='100%' frameborder='0' src='https://monitoring.yandex-team.ru/embed/solomon/mx?q.0.s=%7Bproject%3D%22solomon%22%2C%20cluster%3D%22production%22%2C%20service%3D%22coremon%22%2C%20host%3D%22%3F%3F%3F%22%2C%20sensor%3D%22engine.fileSensors%7Cengine.fileSensorsLimit%22%2C%20projectId%3D%22{project}%22%2C%20shardId%3D%22{shard}%22%7D&normz=off&colors=auto&type=line&interpolation=linear&dsp_method=auto&dsp_aggr=default&dsp_fill=default&range=6mo'}}}}\n\n".format(shard=shard, project=project)

        if hasMemLimits and not isCloud:
          comment += "\ninMemSensorsLimits:\n{{{{iframe height='200' width='100%' frameborder='0' src='https://monitoring.yandex-team.ru/embed/solomon/mx?q.0.s=%7Bproject%3D%22solomon%22%2C%20cluster%3D%22production%22%2C%20service%3D%22coremon%22%2C%20host%3D%22%3F%3F%3F%22%2C%20sensor%3D%22engine.inMemSensors%7Cengine.inMemSensorsLimit%22%2C%20projectId%3D%22{project}%22%2C%20shardId%3D%22{shard}%22%7D&normz=off&colors=auto&type=line&interpolation=linear&dsp_method=auto&dsp_aggr=default&dsp_fill=default&range=6mo'}}}}\n\n".format(shard=shard, project=project)

        comment += "\n"

        data_str = json.dumps(data)

        # This comment is a good idea but does not work, because
        # copying from Startrek history eats up all line breaks
        # which results in all but first commands being commented out
        # commands += "# {shard} (https://solomon.yandex-team.ru/admin/projects/{project}/shards/{shard})\n".format(shard=shard, project=project)

        commands += ("curl --fail -X PUT -H 'Content-Type: application/json' -H \"Authorization: OAuth $SOLOMON_TOKEN\" -d '{data_str}' https://{installation}/api/v2/projects/{project}/shards/{shard} ; test $? -eq 0 && echo -e '\\n\\033[0;32mOK\\033[0m' || echo -e '\\n\\033[0;31mFAIL\\033[0m update quota for shard {shard}' ;\n\n\n".format(installation=installation, data_str=data_str, project=project, shard=shard))

    if len(shards_updates) > 0:
        commands += ("curl --fail -XPOST -H \"Authorization: OAuth $ST_TOKEN\" -H 'Content-Type: application/json' -d '{{\"resolution\": \"fixed\"}}' 'https://st-api.yandex-team.ru/v2/issues/{ticket}/transitions/closed/_execute' | jq 'if type==\"object\" and has(\"statusCode\") then .statusCode else \"success\" end' | grep success ; test $? -eq 0 && echo -e '\\n\\033[0;32mOK\\033[0m closing ticket' || echo -e '\\n\\033[0;31mFAIL\\033[0m closing ticket, retry curl manually to see error message (ticket already closed?)';\n\n".format(ticket=ticket))

    if len(commands) > 0:
        data = {
          "text": "{comment}\n<{{Команды для дежурного:\n%%\n{commands}%%}}>".format(comment=comment, commands=commands)
        }

        token = os.environ["ST_TOKEN"]
        st_headers = {
            "Authorization": "OAuth {token}".format(token=token),
        }

        res = requests.post("https://st-api.yandex-team.ru/v2/issues/{ticket}/comments".format(ticket=ticket), json=data, headers=st_headers)

class HTTPServerV6(HTTPServer):
  address_family = socket.AF_INET6

class MyHandler(BaseHTTPRequestHandler):
     def do_HEAD(s):
         parsed_url = urlparse(s.path)

         if parsed_url.path == "/metrics":
            metrics.handle(s, head=True)
            return

         if parsed_url.path == "/unistat":
            unistat.handle(s, head=True)
            return

         s.send_response(200)
         s.send_header("Content-type", "text/html")
         s.end_headers()

     def do_GET(s):
         parsed_url = urlparse(s.path)

         if parsed_url.path == "/metrics":
            metrics.handle(s)
            return

         if parsed_url.path == "/unistat":
            unistat.handle(s)
            return

         if parsed_url.path == "/alert-webhook":
            s.send_response(200)
            s.send_header("Content-type", "text/html")
            s.end_headers()
            return

         s.send_response(200)
         s.send_header("Content-type", "text/html")
         s.end_headers()

         query = parsed_url.query
         t = [qc.split("=") for qc in query.split("&") if "=" in qc]
         query_components = dict(t)

         error = ""
         warning = ""
         urls = ""
         shard_link = ""

         if 'shard_link' in query_components:
            url = unquote(query_components['shard_link'])
            m = re.search('https?:\/\/solomon\.yandex-team\.ru\/admin\/projects\/([a-zA-Z0-9_-]+)\/shards\/([a-zA-Z0-9-_]+)', url)
            if m is None:
                error = """
<div class="alert alert-warning" role="alert">
  Не получилось распарсить ссылку. Проверьте, что она указана в правильном формате. Пример правильной ссылки: https://solomon.yandex-team.ru/admin/projects/solomon/shards/solomon_production_sys
</div>
"""
            else:
                shard_link = unquote(query_components['shard_link'])
                project = m.group(1)
                shardName = m.group(2)

                try:
                    solomon_api_url = "https://solomon.yandex-team.ru/api/v2/projects/{project}/shards/{shardName}".format(project=project, shardName=shardName)
                    solomon_res = requests.get(solomon_api_url, headers={"Authorization": "OAuth " + os.environ["SOLOMON_TOKEN"]}).json()

                    num_id = solomon_res['numId']

                    template_header = """PRAGMA yt.InferSchema = '1';
use arnold;

-- Autogenerated, different for every shard
$shardId = {num_id};
$tablePath = "home/solomon/VLA_PROD/Metrics";
"""

                    label_cardinality_template = """
--------------------------------------------------------------------------------
-- Label cardinality
--
-- Disclaimer: this data export is not a public API, field sets and names
--             might change in future. No export speed guarantees are provided.
--             Please write to solomon@yandex-team.ru if you need an export
--             with a contract.
--------------------------------------------------------------------------------

select * from (
    select label, count(*) as cnt from (
        select String::SplitToList(labels, '&', true as SkipEmpty) as _labels
        from $tablePath
        where shardId = bitcast($shardId as Uint32)
    )
    flatten list by _labels as label
    group by label
) order by cnt desc;
"""

                    key_cardinality_template = """
--------------------------------------------------------------------------------
-- Key cardinality
--
-- Disclaimer: this data export is not a public API, field sets and names
--             might change in future. No export speed guarantees are provided.
--             Please write to solomon@yandex-team.ru if you need an export
--             with a contract.
--------------------------------------------------------------------------------

select * from (
    select key, count(*) as cnt from (
        select
            String::SplitToList(label, '=')[0] as key
        from (
            select String::SplitToList(labels, '&', true as SkipEmpty) as _labels
            from $tablePath
            where shardId = bitcast($shardId as Uint32)
        )
        flatten list by _labels as label
    ) group by key
) order by cnt desc;
"""

                    host_cardinality_template = """
--------------------------------------------------------------------------------
-- Host cardinality
--
-- Disclaimer: this data export is not a public API, field sets and names
--             might change in future. No export speed guarantees are provided.
--             Please write to solomon@yandex-team.ru if you need an export
--             with a contract.
--------------------------------------------------------------------------------

select * from (
    select value, count(*) as cnt from (
        select
            String::SplitToList(label, '=')[1] as value
        from (
            select String::SplitToList(labels, '&', true as SkipEmpty) as _labels
            from $tablePath
            where shardId = bitcast($shardId as Uint32)
        )
        flatten list by _labels as label
        where String::SplitToList(label, '=')[0] = 'host'
    ) group by value
) order by cnt desc;
"""

                    most_recent_metrics_template = """
--------------------------------------------------------------------------------
-- Most recent metrics
--
-- Disclaimer: this data export is not a public API, field sets and names
--             might change in future. No export speed guarantees are provided.
--             Please write to solomon@yandex-team.ru if you need an export
--             with a contract.
--------------------------------------------------------------------------------

select createdAt, labels from $tablePath
where shardId = bitcast($shardId as Uint32)
order by createdAt desc
limit 1000;
"""

                    template_header = template_header.format(num_id=num_id)

                    label_cardinality = quote(template_header + label_cardinality_template)
                    key_cardinality = quote(template_header + key_cardinality_template)
                    host_cardinality = quote(template_header + host_cardinality_template)
                    most_recent_metrics = quote(template_header + most_recent_metrics_template)

                    urls = """
<div class="row ml-1 mt-5">YQL-запросы:</div>
<div class="row ml-1 mt-2 mb-2">
    <a class="btn btn-info" role="button" target=_blank href="https://yql.yandex-team.ru?query={label_cardinality}">Label cardinality</a></button>
</div>
<div class="row ml-1 mt-2 mb-2">
    <a class="btn btn-info" role="button" target=_blank href="https://yql.yandex-team.ru?query={key_cardinality}">Key cardinality</a></button>
</div>
<div class="row ml-1 mt-2 mb-2">
    <a class="btn btn-info" role="button" target=_blank href="https://yql.yandex-team.ru?query={host_cardinality}">Host cardinality</a></button>
</div>
<div class="row ml-1 mt-2 mb-2">
    <a class="btn btn-info" role="button" target=_blank href="https://yql.yandex-team.ru?query={most_recent_metrics}">Most recent metrics</a></button>
</div>
""".format(label_cardinality=label_cardinality, key_cardinality=key_cardinality, host_cardinality=host_cardinality, most_recent_metrics=most_recent_metrics)

                except Exception as e:
                    print(e)
                    error = """
<div class="alert alert-danger" role="alert">
  Не получилось сгенерировать ссылки. Попробуйте ещё раз или сообщите <a href="https://staff.yandex-team.ru/andreyst">andreyst@</a>.
</div>
"""

         s.wfile.write(bytes("<html><head><meta charset='utf-8'><title>Solomon quotas helper</title><style>{css}</style></head>".format(css=bootstrap.css()), encoding='utf8'))
         body = """
<div class="mt-5 mb-5"></div>
<div class="container">
<div class="jumbotron">
  <h1 class="display-4">Solomon quotas helper</h1>
  {error}
  {warning}
  <p>Укажите ссылку на шард Соломона, чтобы сгенерировать ссылки на YQL-запросы.
  <p>Ссылку на шард можно найти в <a href="https://solomon.yandex-team.ru/admin">админке</a> Соломона: найдите нужный проект, откройте нужный шард на вкладке Shard и скопируйте URL.</p>
  <form action="" method="GET">
  <div class="form-group">
    <label for="shard_link">Ссылка на шард</label>
    <input type="text" class="form-control" id="shard_link" name="shard_link" aria-describedby="emailHelp" value="{shard_link}">
    <small id="emailHelp" class="form-text text-muted">Пример: https://solomon.yandex-team.ru/admin/projects/solomon/shards/solomon_production_sys</small>
  </div>
  <button type="submit" class="btn btn-primary">Сгенерировать ссылки</button>
  </form>

  {urls}
  <div class="mt-5">Вопросы? Пишите на <a href="mailto:solomon@yandex-team.ru">solomon@yandex-team.ru</a>.</div>
</div>
<p><small style="color: gray">Я — крохотный скрипт-хелпер, живущий в Yandex.Deploy. Если мне повезёт, когда-нибудь я вырасту и стану частью большого продакшена, а пока — пожалуйста, не роняйте меня! (✿◠‿◠)</small></p>
</div>


"""
         s.wfile.write(bytes(body.format(error=error, warning=warning, urls=urls, shard_link=shard_link), encoding="utf-8"))
         s.wfile.write(bytes("</body></html>", encoding="utf-8"))

     def do_POST(s):
         s.send_response(200)
         s.send_header("Content-type", "text/html")
         s.end_headers()

         query = urlparse(s.path).query
         print(query)
         content_length = int(s.headers['Content-Length'])
         post_body = s.rfile.read(content_length)

         print(f"New request: {post_body}")
         data = json.loads(post_body.decode("utf-8"))
         print("New ticket {}".format(data["key"]))
         result = re.search("MONITORINGREQ-\d+", data["key"])
         #if result is not None:
         process_ticket(data["key"])
         s.wfile.write(bytes("Request processed", encoding="utf-8"))

if __name__ == '__main__':
    print("Initial supporter: {}".format(supporters[supporters_idx]))

    httpd = HTTPServerV6((HOST_NAME, PORT_NUMBER), MyHandler)
    print(time.asctime(), "Server Starts - %s:%s" % (HOST_NAME, PORT_NUMBER))
    try:
        httpd.serve_forever()
    except KeyboardInterrupt:
        pass
    httpd.server_close()
    print(time.asctime(), "Server Stops - %s:%s" % (HOST_NAME, PORT_NUMBER))
