import os
from urllib import parse

import solomon_api

PUSH_PART_SIZE = 10000

# TODO: set your own parameters to list metrics and range graph data.
PROJECT_ID = "solomon"
OLD_SELECTORS = 'cluster="production", service="gateway", host="cluster", sensor="http.requests.total"'
FROM_TIME = "2018-01-01T00:00:00Z"
TO_TIME = "2021-05-21T00:00:00Z"

# TODO: set your own labels old-to-new mapping.
def mapping(labels):
    new_labels = dict(labels)
    # TODO: implement your own labels mapping.
    return new_labels


client = solomon_api.V2(os.environ["SOLOMON_OAUTH_TOKEN"])


def transform_labels_to_str(labels):
    return ",".join(["%s=\"%s\"" % (key, value) for key, value in labels.items()])


def load_metrics_meta(project_id, selectors):
    url = "/projects/%s/sensors?forceCluster=sas&selectors=%s&pageSize=10000" % (project_id, parse.quote(selectors))
    return client._get_json(url)


def load_stockpile_metrics(project_id, labels_str, from_time, to_time):
    url = "/projects/%s/stockpile/sensor?labels=%s&from=%s&to=%s" % (project_id, parse.quote(labels_str), parse.quote(from_time), parse.quote(to_time))
    data = client._get_json(url, internal=True)
    return data


def transform_to_api_point(point):
    ts = point["tsMillis"] / 1000
    value_num = point["valueNum"]
    value_denom = point["valueDenom"]

    if value_denom == 0 or value_denom == 15000:
        value = value_num
    else:
        value = value_num / value_denom * 1000

    return {
        "ts": ts,
        "value": value
    }


def push_metrics(project_id, labels, kind, points):
    url = "/push?project=%s&cluster=%s&service=%s" % (project_id, labels["cluster"], labels["service"])

    labels_without_shard_key = dict(labels)
    labels_without_shard_key.pop("cluster")
    labels_without_shard_key.pop("service")

    max_points_count = len(points)

    offset = 0

    while offset < max_points_count:
        next_offset = min(offset + PUSH_PART_SIZE, max_points_count)
        print("Part: [%s-%s]/%s" % (offset, next_offset, max_points_count))
        points_part = points[offset:next_offset]

        timeseries = [transform_to_api_point(point) for point in points_part]
        content = {
            "metrics": [{
                "labels": labels_without_shard_key,
                "timeseries": timeseries,
            }]
        }

        try:
            client._post_json(url, content)
        except Exception as e:
            print(e)

        offset += PUSH_PART_SIZE


def process_metrics(project_id, metrics_meta, from_time, to_time):
    metrics = metrics_meta["result"]
    count = 0
    max_count = len(metrics)
    for metric in metrics:
        count += 1
        print("Process metrics: %s/%s" % (count, max_count))

        labels = dict(metric["labels"])
        labels.pop("project")
        labels_str = transform_labels_to_str(labels)

        new_labels = mapping(labels)
        new_labels_str = transform_labels_to_str(new_labels)

        data = load_stockpile_metrics(project_id, labels_str, from_time, to_time)
        kind = data["kind"]
        points = data["points"]

        print("Old labels: %s" % labels_str)
        print("New labels: %s" % new_labels_str)
        print("Kind: %s" % kind)
        print("Points: %s" % len(points))

        push_metrics(project_id, new_labels, kind, points)


def main():
    print("Load metrics meta...")
    metrics_meta = load_metrics_meta(PROJECT_ID, OLD_SELECTORS)
    print("Processing...")
    process_metrics(PROJECT_ID, metrics_meta, FROM_TIME, TO_TIME)
    print("Done")


if __name__ == '__main__':
    main()
