#!/usr/bin/env python3

#
# Tool to parse sensors from stream of Solomon JSON format (see
# https://wiki.yandex-team.ru/solomon/api/dataformat/json/ for more details)
# and outputs them as table(s).
#
# Usage:
#
#   $ cat sensors.json | ./sensors-table.py [[LABELS_TO_FIND]]
#
# You can pass list of label names as free args to output only those sensors
# which have these labels.
#

import json
import sys


# special keys of sensor json
SENSOR_LABELS = "labels"
SENSOR_VALUE = "value"
SENSOR_TIMESERIES = "timeseries"

# special char to concatenate labels in one string
KEY_SEP = ";"


class SensorsInMemTable:
    def __init__(self, labels):
        self._labels = labels   # labels must be sorted
        self._col_width = [len(l) for l in labels]
        self._col_width.append(15)  # sensor value column
        self._rows = []

    def add_sensor(self, sensor):
        new_row = []
        for idx, label in enumerate(self._labels):
            label_value = sensor[SENSOR_LABELS][label]
            self._col_width[idx] = max(self._col_width[idx], len(label_value))
            new_row.append(label_value)
        if SENSOR_VALUE in sensor:
            new_row.append(str(sensor[SENSOR_VALUE]))
        elif SENSOR_TIMESERIES in sensor:
            new_row.append(str(sensor[SENSOR_TIMESERIES]))
        self._rows.append(new_row)

    def output(self, out):
        row_length = sum(self._col_width) + 3 * len(self._col_width)

        for idx, label in enumerate(self._labels + [SENSOR_VALUE]):
            if idx > 0:
                out.write(" | ")
            out.write(label.ljust(self._col_width[idx]))
        out.write("\n")
        out.write("-" * row_length)
        out.write("\n")

        for row in self._rows:
            for idx, col in enumerate(row):
                if idx > 0:
                    out.write(" | ")
                #out.write(col.ljust(self._col_width[idx]))
                out.write(col)
            out.write("\n")
        out.write("-" * row_length)
        out.write("\n\n")


def load_tables(sensors, search_labels):
    tables = {}

    def get_or_create_table(labels):
        table_key = KEY_SEP.join(labels)
        table = tables.get(table_key)
        if table is None:
            table = SensorsInMemTable(labels)
            tables[table_key] = table
        return table

    for sensor in sensors:
        labels = sorted(sensor[SENSOR_LABELS].keys())
        if search_labels is not None:
            if not all(l in labels for l in search_labels):
                continue

        table = get_or_create_table(labels)
        table.add_sensor(sensor)

    return tables


def main():
    sensors = json.load(sys.stdin)["sensors"]
    search_labels = sys.argv[1:] if len(sys.argv) > 1 else None

    tables = load_tables(sensors, search_labels)
    for table in tables.values():
        table.output(sys.stdout)


if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        pass
