#!/usr/bin/env python3

import argparse
import os
import solomon_api
import sys
import urllib.request, urllib.error, urllib.parse


def set_limit(api, project_id, shard_ids, new_limit, dry=True):
    if len(shard_ids) == 0:
        print("Nothing to do: empty list of active shards.")
    for shard_id in shard_ids:
        for limit_type, new_limit_value in new_limit.items():
            try:
                shard = api.get_shard(project_id, shard_id)
                old_limit_value = shard["quotas"][limit_type]

                if old_limit_value >= new_limit_value:
                    print("[SKIP][%s] %s: %d >= %d" % (shard_id, limit_type, old_limit_value, new_limit_value))
                    continue

                log = "%s[%s] %s: %d -> %d" % ("[DRY RUN] " if dry else "", shard_id, limit_type, old_limit_value, new_limit_value)

                if not dry:
                    shard["quotas"][limit_type] = new_limit_value
                    api.put_shard(project_id, shard)

                print(log)

            except urllib.error.HTTPError as e:
                print(shard_id, "not updated:", e)


def is_active(shard):
    return shard["state"] != "INACTIVE"


def main():
    if "OAUTH_TOKEN" not in os.environ:
        print("Please specify your OAuth token in OAUTH_TOKEN " \
                            "environment variable", file=sys.stderr)
        sys.exit(1)

    parser = argparse.ArgumentParser()
    parser.add_argument("-d", "--dry", help="dry run", action='store_true')
    parser.add_argument("-p", "--project", required=True, help="project id")
    parser.add_argument("-s", "--service", help="service id")
    parser.add_argument("-c", "--cluster", help="cluster id")
    parser.add_argument("--sensors-per-url", required=False, type=int,
                        help="maximum sensors per URL")
    parser.add_argument("--max-file-sensors", required=False, type=int,
                        help="maximum file sensors in shard")
    parser.add_argument("shard", nargs="*", help="one or more shard id")

    args = parser.parse_args()
    api = solomon_api.V2(os.environ["OAUTH_TOKEN"])
    dry = args.dry
    
    new_limit = {
        "maxSensorsPerUrl": args.sensors_per_url,
        "maxFileSensors": args.max_file_sensors
    }

    new_limit = dict(filter(lambda x: x[1] is not None, new_limit.items()))

    if args.service:
        shards = api.get_service_shards(args.project, args.service)
        shard_ids = [shard["shardId"] for shard in filter(is_active, shards)]
        set_limit(api, args.project, shard_ids, new_limit, dry)
    elif args.cluster:
        shards = api.get_cluster_shards(args.project, args.cluster)
        shard_ids = [shard["shardId"] for shard in filter(is_active, shards)]
        set_limit(api, args.project, shard_ids, new_limit, dry)
    elif args.shard:
        set_limit(api, args.project, args.shard, new_limit, dry)
    else:
        print("Shards are not specified.", file=sys.stderr)
        print("Use service id or cluster id or explicitly list "\
                            "shard ids", file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()
