#!/usr/bin/env python3

import argparse
import os
import sys

import solomon_api

OAUTH_TOKEN_ENV_VAR = "OAUTH_TOKEN"


def get_project_objects(client, project, object_name):
    print("Project", project)
    print()

    if object_name == "projects":
        try:
            return client._get_json("/projects/" + project)
        except Exception as e:
            print("Failed to load project " + project, e, file=sys.stderr)
            return None

    result = []

    try:
        response = client._get_json("/projects/" + project + "/" + object_name + "?pageSize=10500")
    except Exception as e:
        print("Failed to load project " + project, e, file=sys.stderr)
        return result

    object_items = []
    if 'items' in response:
        object_items = response["items"]
    elif 'result' in response:
        object_items = response["result"]

    for object_item in object_items:
        object_id = object_item["id"]
        try:
            obj = client._get_json("/projects/" + project + "/" + object_name + "/" + object_id)
            result.append(obj)
            print("Load " + object_id + " in " + project)
        except Exception as e:
            print("Failed to load " + object_id + ' in ' + project, e, file=sys.stderr)

    print()

    return result

def update_project(debug_mode, client, project_id, project, newOnlyAuthRead):
    project["onlyAuthRead"] = newOnlyAuthRead
    try:
        if debug_mode:
            print("Set FAKE onlyAuthRead=%s to project %s" % (newOnlyAuthRead, project_id))
        else:
            client._put_json("/projects/" + project_id, project)
            print("Set onlyAuthRead=%s to project %s" % (newOnlyAuthRead, project_id))
        return True
    except Exception as e:
        print("Failed to set onlyAuthRead=%s to project %s" % (newOnlyAuthRead, project_id), e, file=sys.stderr)
        return False


def main():
    if OAUTH_TOKEN_ENV_VAR not in os.environ:
        print("OAuth token isn't specified in", OAUTH_TOKEN_ENV_VAR, "env variable", file=sys.stderr)
        sys.exit(100)

    oauth_token = os.environ[OAUTH_TOKEN_ENV_VAR]

    parser = argparse.ArgumentParser()
    parser.add_argument("--env", default=None, help="environment (CLOUD_PROD|CLOUD_PREPROD)")
    parser.add_argument("--debug", action="store_true", help="enable debug mode")
    parser.add_argument("--revert", action="store_true", help="revert changes")

    args = parser.parse_args()

    revert = args.revert
    debug_mode = args.debug
    env = args.env if args.env is not None else "CLOUD_PROD"

    client = solomon_api.V2(oauth_token, env)

    all_project_items = client._get_json("/projects")

    total_projects_count = len(all_project_items)
    unknown_projects_count = 0
    updated_projects_count = 0
    failed_projects_count = 0
    index = 0

    for cur_project_item in all_project_items:
        index = index + 1
        project_id = cur_project_item["id"]

        print("Project", project_id)

        try:
            project = client._get_json("/projects/" + project_id)
        except Exception as e:
            print("Failed to load project " + project_id, e, file=sys.stderr)
            unknown_projects_count += 1
            continue

        result = None
        is_robot_solomon_project = project["owner"] == 'robot-solomon'
        is_only_auth_read_project = project["onlyAuthRead"] == True

        if is_robot_solomon_project:
            if revert:
                if is_only_auth_read_project:
                    result = update_project(debug_mode, client, project_id, project, False)
            else:
                if not is_only_auth_read_project:
                    result = update_project(debug_mode, client, project_id, project, True)
        if result == True:
            updated_projects_count += 1
        elif result == False:
            failed_projects_count += 1
        print("Stats: ", project_id, "(cur: %s, total: %s, updated: %s, failed: %s, unknown: %s)" % (index, total_projects_count, updated_projects_count, failed_projects_count, unknown_projects_count))

    print("============")
    print("Total stats:")
    print("Total: %s" % total_projects_count)
    print("Updated: %s" % updated_projects_count)
    print("Failed to update: %s" % failed_projects_count)
    print("Unknown: %s" % unknown_projects_count)

if __name__ == "__main__":
    main()
