#!/usr/bin/env python3

# SOLOMON-5410
# Usage:
# pssh run "kikimr db readtable /global/solomon/Config/V2/Service" solomon-gateway-00.svc.cloud.yandex.net > prod_service.csv
# ./set_service_ttl.py prod_service.csv -d -t 10
# ./set_service_ttl.py prod_service.csv -t 10

import os
import sys
import csv
import json
import argparse
import queue
import threading
import urllib.error
import solomon_api


class TtlSetter(threading.Thread):
    def __init__(self, queue, api, dry=False):
        threading.Thread.__init__(self)
        self.queue = queue
        self.api = api
        self.dry = dry

    def run(self):
        while True:
            try:
                service = self.queue.get(timeout=3)
            except queue.Empty:
                return

            self.set_ttl(service)

            self.queue.task_done()

    def set_ttl(self, service):
        service_id = service["id"]
        project_id = service["projectId"]

        ok_log = "{}Seting TTL in project={}, service={}".format(
            "[DRY] " if self.dry else "",
            project_id,
            service_id
        )

        url = "/projects/{}/services/{}".format(project_id, service_id)

        try:
            service_model = self.api._get_json(url)
            service_model["sensorsTtlDays"] = 30
            if not self.dry:
                self.api._put_json(url, service_model)
        except urllib.error.HTTPError as e:
            fail_log = "Failed to set TTL in projetc={}, service={}".format(project_id, service_id)
            print(fail_log)
            print(e.code, e.reason)
            return
        
        print(ok_log)


def main():
    if "OAUTH_TOKEN" not in os.environ:
        print("Please specify your OAuth token in OAUTH_TOKEN environment variable", file=sys.stderr)
        sys.exit(100)

    parser = argparse.ArgumentParser()
    parser.add_argument("infile", type=argparse.FileType("r"))
    parser.add_argument("-d", "--dry", help="dry run", action="store_true")
    parser.add_argument("-t", "--threads", required=True, type=int,
                        help="number of threads")

    args = parser.parse_args()

    # services = json.load(args.infile)
    csvreader = csv.DictReader(args.infile)
    services = [r for r in csvreader]

    pos = 0
    limit = len(services)
    services_without_ttl = queue.Queue()
    for s in services:
        if int(s["sensorsTtlDays"]) == 0 and s["id"].startswith("b1") and "custom" not in s["id"]:
            services_without_ttl.put(s)
            pos += 1
        if pos > limit:
            break

    print(services_without_ttl.qsize())

    api = solomon_api.V2(os.environ["OAUTH_TOKEN"], env="CLOUD_PROD")

    for i in range(args.threads):
        t = TtlSetter(services_without_ttl, api, args.dry)
        t.start()

    services_without_ttl.join()

if __name__ == "__main__":
    main()
