#!/usr/bin/env python3

import os
import sys
import csv
import json
import argparse
import datetime as dt

import urllib.request

import solomon_api

OAUTH_TOKEN_ENV_VAR = "OAUTH_TOKEN"
DATA_API_URL = "/projects/solomon/sensors/data?maxPoints=500"
DAYS = 30


def get_solomon_data_for_days(api, program, days):
    end = dt.datetime.now()
    begin = end - dt.timedelta(days=days)

    data = {
        "program": program, 
        "from": begin.strftime("%Y-%m-%dT%H:%M:%S.%fZ"), 
        "to": end.strftime("%Y-%m-%dT%H:%M:%S.%fZ"),
        "__oldMode": True
    }

    return api._post_json(DATA_API_URL, data)


def get_shards_writes_flow(api, days=7):
    return get_solomon_data_for_days(
        api,
        program='{project="solomon", cluster="production", service="coremon", sensor="engine.pushSensorsToStorage", host="Sas", projectId="kikimr", shardId!="total"}',
        days=days
    )

def get_total_writes_flow(api, days=7):
    return get_solomon_data_for_days(
        api,
        program='drop_nan({project="solomon", cluster="production", service="coremon", sensor="engine.pushSensorsToStorage", host="Sas", projectId="kikimr", shardId="total"})',
        days=days
    )


def get_shards_reads_count(api, days=7):
    return get_solomon_data_for_days(
        api,
        program='{project="solomon", cluster="stockpile_sas", service="stockpile_resources", sensor="stockpile.read.records.rate", projectId="SOLOMON", kind="total", ownerId!="total", producer="total"}',
        days=days
    )


def main():
    if OAUTH_TOKEN_ENV_VAR not in os.environ:
        print("OAuth token isn't specified in", OAUTH_TOKEN_ENV_VAR, "env variable", file=sys.stderr)
        sys.exit(100)

    oauth_token = os.environ[OAUTH_TOKEN_ENV_VAR]
    api = solomon_api.V2(os.environ["SOLOMON_TOKEN"])

    with open("kikimr_shards.json", "r") as f:
        shards = json.load(f)

    shardId_by_ownerId = {}
    for s in shards:
        numId = s.get('numId', 0)
        ownerId = numId if numId > 0 else numId + 2 ** 32
        shardId_by_ownerId[str(ownerId)] = s.get('id', 'N/A')
        

    values = get_total_writes_flow(api, days=DAYS)["vector"][0]["timeseries"]['values']
    total_writes = sum(values) / len(values)

    writes = get_shards_writes_flow(api, days=DAYS)["vector"]

    writes_by_shardId = {}
    for w in writes:
        t = w["timeseries"]
        writes_by_shardId[t["labels"]["shardId"]] = t["__oldStats"]["avg"]

    reads = get_shards_reads_count(api, days=DAYS)["vector"]
    
    reads_by_ownerId = {}
    for r in reads:
        t = r["timeseries"]
        reads_by_ownerId[t["labels"]["ownerId"]] = t["__oldStats"]["avg"]

    with open("kikimr_shards_read_statistics.csv", "w") as csvfile:
        fieldnames = ["ownerId", "shardId", "avg_reads", "avg_writes", "avg_writes_ratio", "shard_url"]
        csvwriter = csv.DictWriter(csvfile, fieldnames=fieldnames)
        csvwriter.writeheader()

        for ownerId in shardId_by_ownerId:
            shardId = shardId_by_ownerId.get(ownerId, "shard_name not found")
            projectId = shardId.split("_")[0]
            avg_writes = writes_by_shardId.get(shardId, "writes count not found")

            try:
                avg_writes_ratio = float(avg_writes) / float(total_writes)
            except ValueError as e:
                avg_writes = avg_writes_ratio = "writes count not found"

        
            csvwriter.writerow({
                "ownerId": ownerId,
                "shardId": shardId,
                "avg_reads": reads_by_ownerId.get(ownerId, 0),
                "avg_writes": avg_writes,
                "avg_writes_ratio": avg_writes_ratio,
                "shard_url": "https://solomon.yandex-team.ru/admin/projects/{}/shards/{}".format(
                    projectId,
                    shardId
                )
            })


if __name__ == "__main__":
    main()
