import urllib.request
import urllib.error
import urllib.parse
import json

from pprint import pprint


class V2:
    API_ENDPOINTS = {
        "PRODUCTION": "https://solomon.yandex-team.ru/api/v2",
        "PRESTABLE": "https://solomon-prestable.yandex-team.ru/api/v2",
        "TESTING": "https://solomon-test.yandex-team.ru/api/v2",
        "CLOUD_PROD": "http://solomon.cloud.yandex-team.ru/api/v2",
        "CLOUD_PREPROD": "http://solomon.cloud-preprod.yandex-team.ru/api/v2",
    }

    def __init__(self, oauth_token, env="PRODUCTION"):
        self._endpoint = self.API_ENDPOINTS[env]
        self._int_endpoint = self._endpoint.replace("v2", "internal")
        self._oauth_token = oauth_token

    def list_projects(self):
        return self._get_json("/projects")

    def delete_project(self, project_id):
        return self._send_delete("/projects/%s" % project_id)

    def get_shard(self, project_id, shard_id):
        return self._get_json("/projects/%s/shards/%s" % (project_id, shard_id))

    def get_service_shards(self, project_id, service_id):
        return self._get_json("/projects/%s/services/%s/clusters" % (
            project_id, service_id))

    def get_cluster_shards(self, project_id, cluster_id):
        return self._get_json("/projects/%s/clusters/%s/services" % (
            project_id, cluster_id))

    def put_shard(self, project_id, shard):
        shard_id = shard["id"]
        path = "/projects/%s/shards/%s" % (project_id, shard_id)
        return self._put_json(path, shard)

    def _get_json(self, path, internal=False):
        headers = {
            "Authorization": "OAuth %s" % self._oauth_token,
            "Accept": "application/json",
        }
        req = urllib.request.Request((self._int_endpoint if internal else self._endpoint) + path, headers=headers)
        resp = urllib.request.urlopen(req)
        return json.load(resp)

    def _put_json(self, path, content):
        data = json.dumps(content)
        headers = {
            "Authorization": "OAuth %s" % self._oauth_token,
            "Accept": "application/json",
            "Content-Type": "application/json",
        }
        req = urllib.request.Request(self._endpoint + path, data.encode("utf-8"), headers)
        req.get_method = lambda: "PUT"
        resp = urllib.request.urlopen(req)
        return json.load(resp)

    def _post_json(self, path, content):
        data = json.dumps(content)
        headers = {
            "Authorization": "OAuth %s" % self._oauth_token,
            "Accept": "application/json",
            "Content-Type": "application/json",
        }
        req = urllib.request.Request(self._endpoint + path, data.encode("utf-8"), headers)
        req.get_method = lambda: "POST"

        resp = ''
        try:
            resp = urllib.request.urlopen(req)
        except Exception as e:
            pprint(e.read())
        return json.load(resp)

    def _send_delete(self, path):
        headers = {
            "Authorization": "OAuth %s" % self._oauth_token,
        }
        req = urllib.request.Request(self._endpoint + path, headers=headers)
        req.get_method = lambda: "DELETE"
        try:
            resp = urllib.request.urlopen(req)
        except Exception as e:
            pprint(e.read())
